create PACKAGE BODY UTILS AS

TYPE VARCHAR2_ARRAY IS TABLE OF VARCHAR2(100);
DT_FORMATS          VARCHAR2_ARRAY; -- Customer : Please Modify the List of Date/Timestamp Formats so that the most common T-SQL literals are at the top
DT_DAY              VARCHAR2_ARRAY; -- Datetime formats starting with Day
DT_MONTH            VARCHAR2_ARRAY; -- Datetime formats starting with Month
DT_YEAR             VARCHAR2_ARRAY; -- Datetime formats starting with Year
DT_TIME             VARCHAR2_ARRAY; -- Datetime formats just having Hour, Minute, Second and Fractional seconds
DT_NLS              VARCHAR2_ARRAY; -- Oracle NLS DateTime formats
DT_TIMESTAMP        VARCHAR2_ARRAY; -- Oracle TIMESTAMP formats

DEFAULT_CHAR_SIZE NUMBER := 30;    

FUNCTION ZERODATETIME RETURN DATE
IS
BEGIN
  RETURN TO_DATE('1900-01-01','yyyy-mm-dd');
END;

FUNCTION LICENSE_ENABLED 
(
  SERVER IN VARCHAR2 DEFAULT 'ase_server' 
) RETURN pls_integer AS 
RET_VAL NUMBER := 0;
BEGIN
  IF 
        (lower(SERVER) = 'ase_server' 
  OR lower(SERVER) = 'ase_ha' 
  OR lower(SERVER) = 'ase_dtm' 
  OR lower(SERVER) = 'ase_java' 
  OR lower(SERVER) = 'ase_asm')
  THEN return 1;
  ELSE return 0 ;
  END IF;
END LICENSE_ENABLED;
-- PRIVATE 
FUNCTION GET_FORMAT_FROM_STYLE(P_STYLE NUMBER) RETURN VARCHAR2
IS
V_FORMAT VARCHAR2(50);
BEGIN
IF DATABASE_TYPE = SYBASE THEN --http://infocenter.sybase.com/help/index.jsp?topic=/com.sybase.help.ase_15.0.blocks/html/blocks/blocks125.htm
V_FORMAT := CASE 

                WHEN P_STYLE = 0   THEN  'Mon DD YYYY HH12:MI AM'
                WHEN P_STYLE = 100 THEN  'Mon DD YYYY HH12:MI AM'
                WHEN P_STYLE = 1   THEN  'MM/DD/YY'
                WHEN P_STYLE = 101 THEN  'MM/DD/YYYY'
                WHEN P_STYLE = 2   THEN  'YY.MM.DD'
                WHEN P_STYLE = 102 THEN  'YYYY.MM.DD'
                WHEN P_STYLE = 3   THEN  'DD/MM/YY'
                WHEN P_STYLE = 103 THEN  'DD/MM/YYYY'
                WHEN P_STYLE = 4   THEN  'DD.MM.YY'
                WHEN P_STYLE = 104 THEN  'DD.MM.YYYY'
                WHEN P_STYLE = 5   THEN  'DD-MM-YY'
                WHEN P_STYLE = 105 THEN  'DD-MM-YYYY'
                WHEN P_STYLE = 6   THEN  'DD Mon YY'
                WHEN P_STYLE = 106 THEN  'DD Mon YYYY'
                WHEN P_STYLE = 7   THEN  'Mon DD, YY'
                WHEN P_STYLE = 107 THEN  'Mon DD, YYYY'
                WHEN P_STYLE = 8   THEN  'HH24:MI:SS'
                WHEN P_STYLE = 108 THEN  'HH24:MI:SS'
                WHEN P_STYLE = 9   THEN  'FMMon  DD YYYY  HH12:MI:SS:FF3AM'
                WHEN P_STYLE = 109 THEN  'FMMon  DD YYYY  HH12:MI:SS:FF3AM'
                WHEN P_STYLE = 10  THEN  'MM-DD-YY'
                WHEN P_STYLE = 110 THEN  'MM-DD-YYYY'
                WHEN P_STYLE = 11  THEN  'YY/MM/DD'
                WHEN P_STYLE = 111 THEN  'YYYY/MM/DD'
                WHEN P_STYLE = 12  THEN  'YYMMDD' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 112 THEN  'YYYYMMDD' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 13  THEN  'YY/DD/MM' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 113 THEN  'YYYY/DD/MM'
                WHEN P_STYLE = 14  THEN  'MM/YY/DD' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 114 THEN  'MM/YYYY/DD' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 15  THEN  'DD/YY/MM'
                WHEN P_STYLE = 115 THEN  'DD/YYYY/MM'
                WHEN P_STYLE = 16  THEN  'Mon  fmDDfm YYYY HH24:MI:SS'
                WHEN P_STYLE = 116 THEN  'Mon  fmDDfm YYYY HH24:MI:SS'
                WHEN P_STYLE = 17  THEN  'FMHH12:MIAM'
                WHEN P_STYLE = 117 THEN  'YYYY/MM/DD HH:MI:SS' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 18  THEN  'HH24:MI'
                WHEN P_STYLE = 118 THEN  'YYYY/MM/DD FMHH12:MIAM' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 19  THEN  'FMHH12:MI:SS:FF3AM'
                WHEN P_STYLE = 119 THEN  'FMHH12:MI:SS:FF3AM'        
                WHEN P_STYLE = 20  THEN  'HH24:MI:SS:FF3'
                WHEN P_STYLE = 120 THEN  'HH24:MI:SS:FF3'
                WHEN P_STYLE = 21  THEN  'YY/MM/DD HH:MI:SS' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 121 THEN  'YYYY/MM/DD HH:MI:SS' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 22  THEN  'YY/MM/DD  fmHH:MIAM'
                WHEN P_STYLE = 122 THEN  'YYYY/MM/DD  fmHH:MIAM'
                WHEN P_STYLE = 23  THEN  'YYYY-MM-DD"T"HH12:MI:SS' -- Different from the doc. Testing suggests this is the actual format
                WHEN P_STYLE = 123 THEN  'YYYY-MM-DD"T"HH12:MI:SS' -- Different from the doc. Testing suggests this is the actual format
                END;
END IF;
IF DATABASE_TYPE = SQLSERVER THEN
V_FORMAT := CASE 
                WHEN P_STYLE = 0   THEN  'MON DD YYYY HH12:MIAM'
                WHEN P_STYLE = 100 THEN  'MON DD YYYY HH12:MIAM'
                WHEN P_STYLE = 1   THEN  'MM/DD/YY'
                WHEN P_STYLE = 101 THEN  'MM/DD/YYYY'
                WHEN P_STYLE = 2   THEN  'YY.MM.DD'
                WHEN P_STYLE = 102 THEN  'YYYY.MM.DD'
                WHEN P_STYLE = 3   THEN  'DD/MM/YY'
                WHEN P_STYLE = 103 THEN  'DD/MM/YYYY'
                WHEN P_STYLE = 4   THEN  'DD.MM.YY'
                WHEN P_STYLE = 104 THEN  'DD.MM.YYYY'
                WHEN P_STYLE = 5   THEN  'DD-MM-YY'
                WHEN P_STYLE = 105 THEN  'DD-MM-YYYY'
                WHEN P_STYLE = 6   THEN  'DD Mon YY'
                WHEN P_STYLE = 106 THEN  'DD Mon YYYY'
                WHEN P_STYLE = 7   THEN  'Mon DD, YY'
                WHEN P_STYLE = 107 THEN  'Mon DD, YYYY'
                WHEN P_STYLE = 8   THEN  'HH24:MI:SS'
                WHEN P_STYLE = 108 THEN  'HH24:MI:SS'
                WHEN P_STYLE = 9   THEN  'FMMon  DD YYYY  HH12:MI:SS:FF3AM'
                WHEN P_STYLE = 109 THEN  'FMMon  DD YYYY  HH12:MI:SS:FF3AM'
                WHEN P_STYLE = 10  THEN  'MM-DD-YY'
                WHEN P_STYLE = 110 THEN  'MM-DD-YYYY'
                WHEN P_STYLE = 11  THEN  'YY/MM/DD'
                WHEN P_STYLE = 111 THEN  'YYYY/MM/DD'
                WHEN P_STYLE = 12  THEN  'YYMMDD'
                WHEN P_STYLE = 112 THEN  'YYYYMMDD'
                WHEN P_STYLE = 13  THEN  'DD Mon YYYY HH24:MI:SS:FF3'
                WHEN P_STYLE = 113 THEN  'DD Mon YYYY HH24:MI:SS:FF3'
                WHEN P_STYLE = 14  THEN  'HH24:MI:SS:FF3'
                WHEN P_STYLE = 114 THEN  'HH24:MI:SS:FF3'
                WHEN P_STYLE = 20  THEN  'YYYY-MM-DD HH24:MI:SS'
                WHEN P_STYLE = 120 THEN  'MM/DD/YY  HH12:MI:SS AM'
                WHEN P_STYLE = 21  THEN  'YYYY-MM-DD HH24:MI:SS.FF3'
                WHEN P_STYLE = 22  THEN  'MM/DD/YY  FMHH12:MI:SS AM'
                WHEN P_STYLE = 122 THEN  'MM/DD/YY  FMHH12:MI:SS AM'
                WHEN P_STYLE = 23  THEN  'YYYY-MM-DD'
                WHEN P_STYLE = 123 THEN  'YYYY-MM-DD'
                WHEN P_STYLE = 121 THEN  'YYYY-MM-DD HH24:MI:SS.FF3'
                WHEN P_STYLE = 126 THEN  'YYYY-MM-DD HH12:MI:SS.FF3'
                WHEN P_STYLE = 127 THEN  'YYYY-MM-DD HH12:MI:SS.FF3'
                WHEN P_STYLE = 130 THEN  'DD Mon YYYY HH12:MI:SS:FF3AM'
                WHEN P_STYLE = 131 THEN  'DD/MM/YY HH12:MI:SS:FF3AM' 
                END;
END IF;

RETURN V_FORMAT;                
END;

FUNCTION CONVERT_STRING_TO_TIMESTAMP (ARG VARCHAR2) RETURN TS
  AS
  BEGIN
   FOR i in DT_FORMATS.FIRST .. DT_FORMATS.LAST
   LOOP
    BEGIN
      RETURN TO_TIMESTAMP(ARG,DT_FORMATS(i));
   EXCEPTION
   WHEN OTHERS THEN
      NULL; -- Keep Trying
    END;
   END LOOP;
   --Attempt to cast one last time, but its really to designed to throw the error to the application when a string is not recognized
   RETURN  TO_TIMESTAMP(ARG,DT_FORMATS(1));
END; 

FUNCTION getNLSTSformat RETURN VARCHAR2
IS
 ts_format VARCHAR2(50);
BEGIN
  SELECT VALUE INTO ts_format FROM V$NLS_PARAMETERS WHERE PARAMETER = 'NLS_TIMESTAMP_FORMAT';
  RETURN ts_format;
END getNLSTSformat;


FUNCTION getNLSDATEformat RETURN VARCHAR2
IS
 dt_format VARCHAR2(50);
BEGIN
  SELECT VALUE INTO dt_format FROM V$NLS_PARAMETERS WHERE PARAMETER = 'NLS_DATE_FORMAT';
  RETURN DT_FORMAT;
END getNLSDATEformat;

FUNCTION PI RETURN NUMBER
IS
    pi NUMBER := 3.141592653589793116;
BEGIN
    RETURN PI;
END PI;

FUNCTION BIGINTTOHEX(P_EXPR NUMBER) RETURN VARCHAR2
IS
    l_number NUMBER;
  BEGIN
    l_number       := P_EXPR;
    -- Need to make sure that we protect against limitation on boundary condition
    IF l_number    <= -2147483647 THEN
      l_number     := -2147483647;
    ELSIF l_number >= 2147483647 THEN
      l_number     := 2147483647;
    END IF;
    RETURN RAWTOHEX(UTL_RAW.CAST_FROM_BINARY_INTEGER(l_number)) ;
END;

FUNCTION BIGINTTOHEX(P_EXPR RAW) RETURN VARCHAR2
IS
BEGIN
  RETURN RAWTOHEX(P_EXPR);
END biginttohex;

FUNCTION BIT_XOR(P_RAW1 RAW   ,P_RAW2 RAW) RETURN RAW
IS
BEGIN
RETURN  UTL_RAW.BIT_XOR(p_raw1,p_raw2);
END;

FUNCTION BIT_XOR(P_NUM1 NUMBER,P_NUM2 NUMBER) RETURN NUMBER
IS BEGIN
RETURN BIT_OR(P_NUM1,P_NUM2) - BITAND(P_NUM1,P_NUM2);
END;

FUNCTION BIT_OR(P_RAW1 RAW    ,P_RAW2 RAW) RETURN RAW
IS BEGIN
RETURN  UTL_RAW.BIT_OR(p_raw1,p_raw2);
END;

FUNCTION BIT_OR(P_NUM1 NUMBER ,P_NUM2 NUMBER) RETURN NUMBER
IS BEGIN
RETURN P_NUM1 + P_NUM2 - BITAND(P_NUM1,P_NUM2);
END;

FUNCTION BIT_AND(P_NUM1 NUMBER   ,P_RAW1 RAW) RETURN RAW
IS BEGIN
RETURN  UTL_RAW.BIT_AND(utl_raw.cast_from_number(p_num1),p_raw1);
END;

FUNCTION BIT_AND(P_RAW1 RAW,P_NUM1 NUMBER) RETURN RAW
IS BEGIN
RETURN UTL_RAW.BIT_AND(p_raw1,utl_raw.cast_from_number(p_num1));
END;


FUNCTION BIT_AND(P_RAW1 RAW   ,P_RAW2 RAW) RETURN RAW
IS BEGIN
RETURN  UTL_RAW.BIT_AND(p_raw1,p_raw2);
END;

FUNCTION BIT_AND(P_NUM1 NUMBER,P_NUM2 NUMBER) RETURN NUMBER 
IS BEGIN
RETURN  BITAND(P_NUM1,P_NUM2);
END;

FUNCTION BIT_COMPLIMENT(P_RAW1 RAW) RETURN RAW
IS BEGIN
  RETURN UTL_RAW.BIT_COMPLEMENT(p_raw1);
END;

FUNCTION BIT_COMPLIMENT(P_NUM1 NUMBER) RETURN NUMBER
IS BEGIN
RETURN  -(P_NUM1 +1);
END;

FUNCTION CONVERT_TO_TIMESTAMP(P_EXPR IN TS	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_TIMESTAMP(P_EXPR IN TSTZ	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN CONVERT_TO_TIMESTAMP(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_TIMESTAMP(P_EXPR IN DATE	                    , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_TIMESTAMP(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS
   v_format VARCHAR2(50);
BEGIN
    IF P_STYLE IS NOT NULL THEN
      V_FORMAT := GET_FORMAT_FROM_STYLE(P_STYLE);
      RETURN TO_TIMESTAMP(P_EXPR, V_FORMAT);
    ELSE
      RETURN CONVERT_STRING_TO_TIMESTAMP(P_EXPR);
    END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END;

FUNCTION CONVERT_TO_DATETIME(P_EXPR IN TS	  , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN CONVERT_TO_TIMESTAMP(P_EXPR,P_PRECISION,P_SCALE,P_STYLE); 
END;

FUNCTION CONVERT_TO_DATETIME	  (P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN CONVERT_TO_DATETIME(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

FUNCTION CONVERT_TO_DATETIME(P_EXPR IN DATE	  , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS BEGIN
  RETURN CONVERT_TO_TIMESTAMP(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_DATETIME(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN TS
IS
BEGIN
  RETURN CONVERT_TO_TIMESTAMP(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_DATE	  (P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
   RETURN CONVERT_TO_DATE(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

FUNCTION CONVERT_TO_DATE	  (P_EXPR IN TS              , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE));
END;

FUNCTION CONVERT_TO_DATE	  (P_EXPR IN DATE	                    , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE));
END;

FUNCTION CONVERT_TO_DATE	  (P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE));
END;



FUNCTION CONVERT_TO_SMALLDATETIME	  (P_EXPR IN TS              , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE),'MI');
END;

FUNCTION CONVERT_TO_SMALLDATETIME	  (P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
   RETURN CONVERT_TO_SMALLDATETIME(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

FUNCTION CONVERT_TO_SMALLDATETIME	  (P_EXPR IN DATE	                    , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE),'MI');
END;

FUNCTION CONVERT_TO_SMALLDATETIME	  (P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN DATE
IS BEGIN
  RETURN TRUNC(CONVERT_TO_DATETIME(P_EXPR,P_PRECISION,P_SCALE,P_STYLE),'MI');
END;

FUNCTION CONVERT_TO_NUMBER	(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN NUMBER
IS BEGIN
RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_NUMBER	(P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN NUMBER
IS BEGIN
RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_FLOAT	  (P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN FLOAT
IS BEGIN
  return P_EXPR;
END;

FUNCTION CONVERT_TO_FLOAT	  (P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN FLOAT
IS BEGIN
  return p_expr;
END;

FUNCTION CONVERT_TO_FLOAT	  (P_EXPR IN FLOAT  	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN FLOAT
IS BEGIN
  RETURN P_EXPR;
END;

---
FUNCTION CONVERT_TO_VARCHAR2	(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN VARCHAR2
IS BEGIN
  RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_VARCHAR2	(P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN VARCHAR2
IS BEGIN
  RETURN P_EXPR;
END;

FUNCTION CONVERT_TO_VARCHAR2	(P_EXPR IN DATE		                  , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN VARCHAR2
IS
   v_format VARCHAR2(50);
   v_result VARCHAR2(100);
BEGIN
  IF P_STYLE IS NOT NULL THEN
    V_FORMAT := GET_FORMAT_FROM_STYLE(P_STYLE);
    v_result := TO_CHAR(P_EXPR,V_FORMAT);  
  ELSE
    v_result := TO_CHAR(P_EXPR);
  END IF;
  IF P_PRECISION <> 0 THEN
    v_result := SUBSTR(v_result,1,p_precision);
  END IF;
  RETURN v_result;
END;

FUNCTION CONVERT_TO_VARCHAR2	(P_EXPR IN TS		            , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN VARCHAR2
IS
   v_format VARCHAR2(50);
BEGIN
  IF P_STYLE IS NOT NULL THEN
    V_FORMAT := GET_FORMAT_FROM_STYLE(P_STYLE);
    RETURN TO_CHAR(P_EXPR,V_FORMAT);  
  ELSE
    RETURN TO_CHAR(P_EXPR);
  END IF;
END;

FUNCTION CONVERT_TO_VARCHAR2	(P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN VARCHAR2
IS BEGIN
   RETURN CONVERT_TO_VARCHAR2(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

FUNCTION CONVERT_TO_NUMBER(P_BLOB_EXPR IN BLOB, P_PRECISION IN NUMBER DEFAULT 0, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN NUMBER
IS
  v_expr RAW(32767);
BEGIN
    v_expr := DBMS_LOB.SUBSTR(P_BLOB_EXPR);
    -- v_expr := REGEXP_REPLACE(v_expr, '0x', '', 1, 1, 'i');
    RETURN UTL_RAW.CAST_TO_BINARY_INTEGER (v_expr, 3);  -- where 3 is machine_endian
END;

FUNCTION CONVERT_TO_BLOB(P_EXPR IN NUMBER, P_PRECISION IN NUMBER DEFAULT 0, P_SCALE IN NUMBER DEFAULT 0, P_STYLE IN NUMBER DEFAULT 0) RETURN BLOB
IS
BEGIN
   RETURN UTL_RAW.CAST_FROM_BINARY_INTEGER(P_EXPR, 3); -- where 3 is machine_endian
END;

FUNCTION CONVERT_TO_VARCHAR2(P_EXPR IN BLOB, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT 0) RETURN VARCHAR2
IS
  v_expr RAW(32767);
BEGIN
   v_expr := DBMS_LOB.SUBSTR(P_EXPR);
   RETURN UTL_RAW.CAST_TO_VARCHAR2(v_expr);
END;

FUNCTION CONVERT_TO_BLOB(P_EXPR IN VARCHAR2, P_PRECISION IN NUMBER DEFAULT 0, P_SCALE IN NUMBER DEFAULT 0, P_STYLE IN NUMBER DEFAULT 0) RETURN BLOB
IS
BEGIN
   RETURN UTL_RAW.CAST_TO_RAW(P_EXPR);
END;

FUNCTION SUBSTR_(P_EXPR IN VARCHAR2,P_PRECISION IN NUMBER DEFAULT NULL) RETURN CHAR
AS
BEGIN
  IF P_PRECISION IS  NULL THEN
    RETURN SUBSTR(P_EXPR,1,DEFAULT_CHAR_SIZE);
  ELSE
    RETURN SUBSTR(P_EXPR,1,P_PRECISION);
  END IF;
END;
--- MASTER CONVERT_TO_CHAR for DATETIME values
FUNCTION CONVERT_TO_CHAR_(P_EXPR IN TS	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS
V_FORMAT VARCHAR2(100);
V_RESULT VARCHAR2(4000);
BEGIN
  IF P_STYLE IS NOT NULL THEN
    V_FORMAT := GET_FORMAT_FROM_STYLE(P_STYLE);
    v_result := TO_CHAR(P_EXPR,V_FORMAT);  
  ELSE
    v_result := TO_CHAR(P_EXPR);
  END IF;
  RETURN SUBSTR_(v_result,P_PRECISION);
END;
---
FUNCTION CONVERT_TO_CHAR	(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS BEGIN
     RETURN SUBSTR_(P_EXPR,P_PRECISION);
END;

FUNCTION CONVERT_TO_CHAR	(P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT NULL  , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS BEGIN
  RETURN  SUBSTR_(P_EXPR,P_PRECISION);
END;

FUNCTION CONVERT_TO_CHAR	(P_EXPR IN DATE		                  , P_PRECISION IN NUMBER DEFAULT NULL   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS
   v_format VARCHAR2(50);
   v_result CHAR(4000);
BEGIN
   RETURN CONVERT_TO_CHAR_(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_CHAR	(P_EXPR IN TS		            , P_PRECISION IN NUMBER DEFAULT NULL   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS
   v_format VARCHAR2(50);
BEGIN
   RETURN CONVERT_TO_CHAR_(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_CHAR	(P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CHAR
IS BEGIN
   RETURN CONVERT_TO_CHAR(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

FUNCTION CONVERT_TO_NVARCHAR2	(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN NVARCHAR2
IS
BEGIN
   RETURN UTL_RAW.CAST_TO_NVARCHAR2(UTL_RAW.CAST_TO_RAW(CONVERT_TO_VARCHAR2(P_EXPR, P_PRECISION, P_SCALE, P_STYLE))); 
END;

FUNCTION CONVERT_TO_NVARCHAR2	(P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN NVARCHAR2
IS
BEGIN
   RETURN UTL_RAW.CAST_TO_NVARCHAR2(UTL_RAW.CAST_TO_RAW(CONVERT_TO_VARCHAR2(P_EXPR, P_PRECISION, P_SCALE, P_STYLE))); 
END;

FUNCTION CONVERT_TO_NVARCHAR2	(P_EXPR IN DATE		                  , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN NVARCHAR2
IS
BEGIN
   RETURN UTL_RAW.CAST_TO_NVARCHAR2(UTL_RAW.CAST_TO_RAW(CONVERT_TO_VARCHAR2(P_EXPR, P_PRECISION, P_SCALE, P_STYLE)));
END;   

FUNCTION CONVERT_TO_NVARCHAR2	(P_EXPR IN TS		            , P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN NVARCHAR2
IS
BEGIN
  RETURN UTL_RAW.CAST_TO_NVARCHAR2(UTL_RAW.CAST_TO_RAW(CONVERT_TO_VARCHAR2(P_EXPR, P_PRECISION, P_SCALE, P_STYLE)));
END;

FUNCTION CONVERT_TO_NVARCHAR2	(P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN NVARCHAR2
IS BEGIN
   RETURN CONVERT_TO_NVARCHAR2(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

--- MASTER CONVERT_TO_CLOB for DATETIME values
FUNCTION CONVERT_TO_CLOB_(P_EXPR IN TS	, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS
V_FORMAT VARCHAR2(100);
V_RESULT VARCHAR2(4000);
BEGIN
  IF P_STYLE IS NOT NULL THEN
    V_FORMAT := GET_FORMAT_FROM_STYLE(P_STYLE);
    v_result := TO_CHAR(P_EXPR,V_FORMAT);  
  ELSE
    v_result := TO_CHAR(P_EXPR);
  END IF;
  
  RETURN SUBSTR_(v_result,P_PRECISION);
END;

FUNCTION CONVERT_TO_CLOB	(P_EXPR IN TSTZ	, P_PRECISION IN NUMBER DEFAULT NULL, P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS BEGIN
   RETURN CONVERT_TO_CLOB(CAST(P_EXPR AS TS),P_PRECISION,P_SCALE,P_STYLE); --REMOVE TZ
END;

---
FUNCTION CONVERT_TO_CLOB	(P_EXPR IN CLOB	                , P_PRECISION IN NUMBER DEFAULT 4000, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS BEGIN
     RETURN SUBSTR_(P_EXPR,P_PRECISION);
END;

FUNCTION CONVERT_TO_CLOB	(P_EXPR IN VARCHAR2	                , P_PRECISION IN NUMBER DEFAULT 4000, P_SCALE IN NUMBER DEFAULT NULL, P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS BEGIN
     RETURN SUBSTR_(P_EXPR,P_PRECISION);
END;

FUNCTION CONVERT_TO_CLOB	(P_EXPR IN NUMBER  	                , P_PRECISION IN NUMBER DEFAULT 4000 , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS BEGIN
  RETURN  SUBSTR_(P_EXPR,P_PRECISION);
END;

FUNCTION CONVERT_TO_CLOB	(P_EXPR IN DATE		                  , P_PRECISION IN NUMBER DEFAULT NULL   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS
   v_format VARCHAR2(50);
   v_result CHAR(4000);
BEGIN
   RETURN CONVERT_TO_CLOB_(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;

FUNCTION CONVERT_TO_CLOB	(P_EXPR IN TS		            , P_PRECISION IN NUMBER DEFAULT NULL   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT NULL) RETURN CLOB
IS
   v_format VARCHAR2(50);
BEGIN
   RETURN CONVERT_TO_CLOB_(P_EXPR,P_PRECISION,P_SCALE,P_STYLE);
END;


-- commented out for now as it returns null
--FUNCTION CONVERT_TO_CHAR(P_EXPR IN BLOB, P_PRECISION IN NUMBER DEFAULT 0   , P_SCALE IN NUMBER DEFAULT 0   , P_STYLE IN NUMBER DEFAULT 0) RETURN CHAR
--IS
--BEGIN
--     RETURN SUBSTR_(CONVERT_TO_VARCHAR2(P_EXPR, P_PRECISION, P_SCALE, P_STYLE), P_PRECISION);
--END;

FUNCTION datename_(p_part_expr IN VARCHAR2, p_date_expr IN TS)
RETURN VARCHAR2
IS
  v_part VARCHAR2(15);
  v_timestamp TS := p_date_expr;
  v_wkday VARCHAR2(10);
  v_year VARCHAR2(4);
  v_temp VARCHAR2(30);
BEGIN
      v_part := UPPER(p_part_expr);
      IF v_part IN ('YEAR', 'YY', 'YYYY') THEN RETURN TO_CHAR(v_timestamp, 'YYYY');
      ELSIF v_part IN ('QUARTER', 'QQ', 'Q') THEN RETURN TO_CHAR(v_timestamp, 'Q');
      ELSIF v_part IN ('MONTH', 'MM', 'M') THEN RETURN TO_CHAR(v_timestamp, 'Month');
      ElSIF v_part IN ('DAYOFYEAR', 'DY', 'Y') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'DDD'));
      ELSIF v_part IN ('DAY', 'DD', 'D') THEN RETURN TO_CHAR(v_timestamp, 'DD');
      ELSIF v_part IN ('WEEKDAY', 'DW', 'W') THEN RETURN TO_CHAR(v_timestamp, 'Day');
      ElSIF v_part IN ('WEEK', 'WK', 'WW') THEN
         -- commented out this code for now needs to be reviewed if the
         -- result is not consistent with sybase function call datename(wk,getdate())
         -- Oracle returns 1 short when compared to Sybase so adding 1 to the result.
         /*
         v_year := TO_CHAR(v_timestamp, 'YYYY');
         v_wkday := TO_CHAR(TO_DATE('01-01-'|| v_year, 'MM-DD-YYYY'), 'DAY');
         IF v_wkday = TO_CHAR(v_timestamp, 'DAY') THEN
            RETURN TO_CHAR(v_timestamp, 'WW');
         ELSE
            RETURN TO_CHAR(v_timestamp, 'WW') + 1;
         END IF;
         */
         RETURN TO_CHAR(v_timestamp, 'WW');
      ElSIF v_part IN ('HOUR', 'HH') THEN RETURN TO_CHAR(v_timestamp, 'fmHH24');
      ElSIF v_part IN ('MINUTE', 'MI', 'N') THEN RETURN TO_CHAR(v_timestamp, 'fmMI');
      ElSIF v_part IN ('SECOND', 'SS', 'S') THEN RETURN TO_CHAR(v_timestamp, 'fmSS');
      ElSIF v_part IN ('MILLISECOND', 'MS', 'FF3') THEN 
          v_temp := TO_CHAR(v_timestamp, 'FF3');
          IF v_temp = '000' THEN
             RETURN '0';
          ELSE   
             RETURN v_temp;
          END IF;   
      ElSIF v_part IN ('MICROSECOND', 'MCS', 'FF6') THEN 
          v_temp := TO_CHAR(v_timestamp, 'FF6');
          IF v_temp = '000000' THEN
             RETURN '0';
          ELSE   
             RETURN v_temp;
          END IF;
      ElSIF v_part IN ('NANOSECOND', 'NS', 'FF9') THEN 
          v_temp := TO_CHAR(v_timestamp, 'FF9');
          IF v_temp = '000000000' THEN
             RETURN '0';
          ELSE   
             RETURN v_temp;
          END IF;
      ElSIF v_part IN ('TZOFFSET', 'TZ') THEN RETURN TO_CHAR(v_timestamp, 'TZH') || ':' || TO_CHAR(v_timestamp, 'TZM');
      ELSE
        RETURN NULL;
      END IF;

EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END datename_;

FUNCTION datename(p_part_expr IN VARCHAR2, p_date_str IN VARCHAR2) 
RETURN VARCHAR2
IS
    v_part VARCHAR2(15);
    v_date DATE;
    v_ts TS;
    v_wkday VARCHAR2(10);
    v_year VARCHAR2(4);
BEGIN
      v_ts := CONVERT_STRING_TO_TIMESTAMP(TRIM(p_date_str));   
      RETURN datename_(p_part_expr, v_TS);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END datename;

FUNCTION datename(p_part_expr IN VARCHAR2, p_date_expr IN DATE) 
RETURN VARCHAR2
IS
BEGIN
    RETURN datename_(p_part_expr, p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK); 
END datename;

FUNCTION datename(p_part_expr IN VARCHAR2, p_date_expr IN TS) 
RETURN VARCHAR2
IS
BEGIN
   RETURN datename_(p_part_expr, p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK); 
END datename;

FUNCTION DATENAME(P_PART_EXPR IN VARCHAR2, P_DATE_EXPR IN TSTZ)  RETURN VARCHAR2
IS BEGIN
   RETURN DATENAME(P_PART_EXPR , CAST(P_DATE_EXPR AS TS)); --REMOVE TZ
END;


FUNCTION datepart_(p_part_expr IN VARCHAR2, p_date_expr IN TS)
RETURN NUMBER
IS
  v_part VARCHAR2(15) := p_part_expr;
  v_timestamp TS := p_date_expr;
  v_wkday VARCHAR2(10);
  v_year VARCHAR2(4);
BEGIN
      v_part := UPPER(p_part_expr);
      IF v_part IN ('YEAR', 'YY', 'YYYY') THEN  RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'YYYY'));
      ELSIF v_part IN ('QUARTER', 'QQ', 'Q')  THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'Q'));
      ELSIF v_part IN ('MONTH', 'MM', 'M') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'MM'));
      ElSIF v_part IN ('DAYOFYEAR', 'DY', 'Y') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'DDD'));
      ELSIF v_part IN ('DAY', 'DD', 'D') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'DD'));
      ELSIF v_part IN ('WEEKDAY', 'DW', 'W') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'D'));
      -- Oracle returns 1 short when compared to Sybase so adding 1 to the result.
      ElSIF v_part IN ('WEEK', 'WK', 'WW') THEN  
         v_year := TO_CHAR(v_timestamp, 'YYYY');
         FOR i in DT_FORMATS.FIRST .. DT_FORMATS.LAST
         LOOP
            BEGIN
              v_wkday := TO_CHAR(TO_DATE('01-01-'|| v_year, DT_FORMATS(i)), 'DAY');
              EXIT;
            EXCEPTION
             WHEN OTHERS THEN
                NULL; 
            END;
         END LOOP;
         IF v_wkday = TO_CHAR(v_timestamp, 'DAY') THEN
            RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'WW'));
         ELSE
            RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'WW')) + 1;
         END IF; 
      ElSIF v_part IN ('HOUR', 'HH') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'HH24'));
      ElSIF v_part IN ('MINUTE', 'MI', 'N') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'MI'));
      ElSIF v_part IN ('SECOND', 'SS', 'S') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'SS'));
      ElSIF v_part IN ('MILLISECOND', 'MS', 'FF3') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'FF3'));
      ElSIF v_part IN ('MICROSECOND', 'MCS', 'US', 'FF6') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'FF6'));
      ElSIF v_part IN ('NANOSECOND', 'NS', 'FF9') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'FF9'));
      ElSIF v_part IN ('CALYEAROFWEEK', 'CYR', 'IYYY') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'IYYY')); 
      ElSIF v_part IN ('CALWEEKOFYEAR', 'CWK', 'IW') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'IW'));
      -- Oracle returns 1 more when compared to Sybase so subtract 1 to the result.
      ElSIF v_part IN ('CALDAYOFWEEK', 'CDW', 'D') THEN RETURN TO_NUMBER(TO_CHAR(v_timestamp, 'D')) - 1;
      ELSE
        RETURN NULL;
      END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_BACKTRACE);
END datepart_;

FUNCTION datepart(p_part_expr IN VARCHAR2, p_date_str IN VARCHAR2)
RETURN NUMBER
IS
   v_ts TS;
BEGIN  
      v_ts := CONVERT_STRING_TO_TIMESTAMP(p_date_str);
      RETURN datepart_(p_part_expr, v_ts);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_BACKTRACE);
END datepart;

FUNCTION datepart(p_part_expr IN VARCHAR2, p_date_expr IN TS)
RETURN NUMBER
IS
BEGIN
    RETURN datepart_(p_part_expr, p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datepart;

FUNCTION DATEPART(P_PART_EXPR IN VARCHAR2, P_DATE_EXPR IN TSTZ) RETURN NUMBER
IS BEGIN
   RETURN DATEPART(P_PART_EXPR , CAST(P_DATE_EXPR AS TS)); --REMOVE TZ
END;

FUNCTION datepart(p_part_expr IN VARCHAR2, p_date_expr IN DATE)
RETURN NUMBER
IS
BEGIN
    RETURN datepart_(p_part_expr, p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datepart;


FUNCTION dateadd_(p_interval IN VARCHAR2,  p_interval_val IN NUMBER, p_date_exp IN TS)
RETURN TS
IS
  v_ucase_interval VARCHAR2(10);
  v_date TS;
  v_datestr VARCHAR2(30);
  v_result TS;
BEGIN
    v_date := p_date_exp;
    v_ucase_interval := UPPER(p_interval);    
    IF v_ucase_interval IN ('YEAR', 'YY', 'YYYY') 
    THEN
      RETURN ADD_MONTHS(v_date, p_interval_val * 12);
    ELSIF v_ucase_interval IN ('QUARTER', 'QQ', 'Q') 
    THEN
      IF LAST_DAY(v_date) = v_date THEN
         v_datestr := EXTRACT(MONTH FROM v_date) + (p_interval_val * 3);
         v_datestr := v_datestr || '-' || EXTRACT(DAY FROM v_date) || '-' || EXTRACT(YEAR FROM v_date);
         v_datestr := v_datestr || ' ' || TO_CHAR(v_date, 'HH12') || ':' || TO_CHAR(v_date, 'MI') || ':' || TO_CHAR(v_date, 'SS');
         v_datestr := v_datestr || '.' || TO_CHAR(v_date, 'FF3 AM') ;
         v_result := TO_TIMESTAMP(v_datestr, 'MM-DD-YYYY HH12:MI:SS.FF3 AM');
         RETURN v_result;
      ELSE
         RETURN ADD_MONTHS(v_date, p_interval_val * 3);
      END IF;   
    ELSIF v_ucase_interval IN ('MONTH', 'MM', 'M') 
    THEN
      BEGIN
         v_result := v_date + NUMTOYMINTERVAL(p_interval_val, 'MONTH') + NUMTODSINTERVAL(0, 'HOUR');
      EXCEPTION WHEN OTHERS THEN
       --problem due to resulting date not being able to handle a particular day like (Ex: Feb 31)
       v_result := ADD_MONTHS(v_date, p_interval_val) ; -- this will use the last valid day (Ex: Feb 28)
      END;
      RETURN v_result;
    ElSIF v_ucase_interval IN ('DAYOFYEAR', 'DY', 'Y', 'DAY', 'DD', 'D', 'WEEKDAY', 'DW', 'W') 
    THEN
      RETURN v_date + NUMTODSINTERVAL(p_interval_val, 'DAY');
    ElSIF v_ucase_interval IN ('WEEK', 'WK', 'WW') 
    THEN
      RETURN v_date + (p_interval_val * 7);
    ElSIF v_ucase_interval IN ('HOUR', 'HH') 
    THEN
      -- RETURN v_date + (p_interval_val / 24);
       RETURN v_date + NUMTODSINTERVAL(p_interval_val, 'HOUR');
    ElSIF v_ucase_interval IN ('MINUTE', 'MI', 'N') 
    THEN
      -- RETURN v_date + NUMTODSINTERVAL((p_interval_val / 24 / 60), 'MINUTE');
      RETURN v_date + NUMTODSINTERVAL(p_interval_val, 'MINUTE');
    ElSIF v_ucase_interval IN ('SECOND', 'SS', 'S') 
    THEN
      -- RETURN v_date + (p_interval_val / 24 / 60 / 60);
      RETURN v_date + NUMTODSINTERVAL(p_interval_val, 'SECOND');
    ElSIF v_ucase_interval IN ('MILLISECOND', 'MS') 
    THEN
      -- result accurate to one three-hundredth of a second 
      RETURN v_date + NUMTODSINTERVAL(3.33 * ROUND(p_interval_val/3.33), 'SECOND')/1000;
    ELSE
      RETURN NULL;
    END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK); 
END dateadd_;

FUNCTION dateadd(p_interval IN VARCHAR2, p_interval_val IN NUMBER, p_date_str IN VARCHAR2)
RETURN TS
IS
    v_ts TS;
BEGIN
    v_ts := CONVERT_STRING_TO_TIMESTAMP(p_date_str);
    RETURN dateadd_(p_interval, p_interval_val, v_ts);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END dateadd;

FUNCTION dateadd(p_interval IN VARCHAR2, p_interval_val IN NUMBER, p_date_expr IN DATE)
RETURN DATE
IS
BEGIN
    RETURN CAST(dateadd_(p_interval, p_interval_val, p_date_expr) AS DATE);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END dateadd;

FUNCTION dateadd(p_interval IN VARCHAR2, p_interval_val IN NUMBER, p_date_expr IN TS)
RETURN TS
IS  
BEGIN
    RETURN dateadd_(p_interval, p_interval_val, p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END dateadd;

FUNCTION DATEADD(P_INTERVAL IN VARCHAR2, P_INTERVAL_VAL IN NUMBER, P_DATE_EXPR IN TSTZ) RETURN TS
IS BEGIN
   RETURN DATEADD(P_INTERVAL ,P_INTERVAL_VAL, CAST(P_DATE_EXPR AS TS)); --REMOVE TZ
END;

FUNCTION degrees(p_angle_radians IN NUMBER) 
RETURN NUMBER
IS
BEGIN
    IF p_angle_radians IS NULL THEN
      RETURN NULL;
    END IF;
    
    RETURN ROUND(p_angle_radians / pi() * 180);
EXCEPTION 
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END degrees;

--SQLSERVER
FUNCTION year_(p_date_str IN VARCHAR2)
RETURN NUMBER
IS
    v_date DATE;
BEGIN
    v_date := CONVERT_STRING_TO_TIMESTAMP(p_date_str);
    IF v_date IS NULL THEN
      RETURN NULL;
    END IF;
    RETURN TO_NUMBER(TO_CHAR(v_date, 'YYYY'));
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END year_;

FUNCTION year_(p_date_expr IN DATE) 
RETURN NUMBER
IS
BEGIN
   RETURN EXTRACT(YEAR FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END year_;

FUNCTION year_(p_date_expr IN TS)
RETURN NUMBER
IS
BEGIN
   RETURN EXTRACT(YEAR FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END year_;

FUNCTION YEAR_(P_DATE_EXPR   IN TSTZ) RETURN NUMBER
IS BEGIN
   RETURN YEAR_(CAST(P_DATE_EXPR AS TS)); --REMOVE TZ
END;

FUNCTION difference(p_expr1 IN VARCHAR2, p_expr2 IN VARCHAR2)
RETURN NUMBER
IS
    sound_ex_val_1 CHAR(4);
    sound_ex_val_2 CHAR(4);
    similarity NUMBER := 0;
    idx NUMBER := 1; 
BEGIN
    IF p_expr1 IS NULL OR p_expr2 IS NULL THEN
       RETURN NULL;
    END IF; 
    sound_ex_val_1 := SOUNDEX(p_expr1);
    sound_ex_val_2 := SOUNDEX(p_expr2); 
    LOOP
       IF SUBSTR(sound_ex_val_1, idx, 1) = SUBSTR(sound_ex_val_2, idx, 1) THEN
          similarity := similarity + 1;
       END IF;
       
       idx := idx + 1;   
       EXIT WHEN idx > 4;
    END LOOP;
    RETURN similarity;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END difference;

FUNCTION datediff_sqlserver(p_datepart IN VARCHAR2, p_start_date_expr IN TS, p_end_date_expr IN TS)
RETURN NUMBER
IS
    v_ret_value NUMBER := NULL;
    v_part VARCHAR2(15);
    v_start_ts  TS;
    v_end_ts  TS;
    v_start_date  DATE;
    v_end_date  DATE;
BEGIN
    v_part := p_datepart;
    v_start_ts := p_start_date_expr;
    v_end_ts := p_end_date_expr;
    v_start_date := TO_DATE(EXTRACT(YEAR FROM v_start_ts) || '-' || EXTRACT(MONTH FROM v_start_ts) || '-' || EXTRACT(DAY FROM v_start_ts), 'YYYY-MM-DD');
    v_end_date := TO_DATE(EXTRACT(YEAR FROM v_end_ts) || '-' || EXTRACT(MONTH FROM v_end_ts) || '-' || EXTRACT(DAY FROM v_end_ts), 'YYYY-MM-DD');
    v_part := UPPER(p_datepart);
    
    IF v_part IN ('YEAR', 'YY', 'YYYY') THEN
      v_ret_value :=EXTRACT(YEAR FROM v_end_ts) - EXTRACT(YEAR FROM v_start_ts) ;
    ELSIF v_part IN ('QUARTER', 'QQ', 'Q') THEN
       v_ret_value := ROUND(MONTHS_BETWEEN(v_end_ts, v_start_ts) / 3);
    ELSIF v_part IN ('MONTH', 'MM', 'M') THEN
       v_ret_value := ROUND(MONTHS_BETWEEN(TRUNC(v_end_ts, 'MM'), TRUNC(v_start_ts, 'MM')));
     ElSIF v_part IN ('DAYOFYEAR', 'DY', 'Y') THEN
       v_ret_value := ROUND(CAST(v_end_ts AS DATE) - CAST(v_start_ts AS DATE));
    ElSIF v_part IN ('DAY', 'DD', 'D') THEN
       v_ret_value := ROUND(v_end_date - v_start_date);
    ElSIF v_part IN ('WEEK', 'WK', 'WW') THEN
       v_ret_value := ROUND((CAST(v_end_ts AS DATE) - CAST(v_start_ts AS DATE)) / 7);
    ELSIF v_part IN ('WEEKDAY', 'DW', 'W') THEN
       -- v_ret_value := TO_CHAR(v_end_ts, 'D') - TO_CHAR(v_start_ts, 'D');
       -- In Sybase the result is similar to using date part 'Week 'or 'wk' in datediff function
       IF EXTRACT(YEAR FROM v_end_ts) = EXTRACT(YEAR FROM v_start_ts) THEN
          v_ret_value := EXTRACT(DAY FROM v_end_ts) - EXTRACT(DAY FROM v_start_ts);
       ELSE
          v_ret_value := ROUND((TRUNC(v_end_ts, 'DD') - TRUNC(v_start_ts, 'DD')) / 7);
       END IF;
    ElSIF v_part IN ('HOUR', 'HH') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24;
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts))));
    ElSIF v_part IN ('MINUTE', 'MI', 'N') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24 * 60;
       v_ret_value := v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts)) * 60);
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(MINUTE FROM v_end_ts) - EXTRACT(MINUTE FROM v_start_ts))));
    ElSIF v_part IN ('SECOND', 'SS', 'S') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24 * 60 * 60;
       v_ret_value := v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts)) * 60 * 60);
       v_ret_value := v_ret_value + ((EXTRACT(MINUTE FROM v_end_ts) - EXTRACT(MINUTE FROM v_start_ts)) * 60);
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(SECOND FROM v_end_ts) - EXTRACT(SECOND FROM v_start_ts))));
    ElSIF v_part IN ('MILLISECOND', 'MS') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24 * 60 * 60 * 1000;
       v_ret_value := v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts)) * 60 * 60 * 1000);
       v_ret_value := v_ret_value + ((EXTRACT(MINUTE FROM v_end_ts) - EXTRACT(MINUTE FROM v_start_ts)) * 60 * 1000);
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(SECOND FROM v_end_ts) - EXTRACT(SECOND FROM v_start_ts)) * 1000));
    ElSIF v_part IN ('MICROSECOND', 'MCS') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24 * 60 * 60 * 1000000;
       v_ret_value := v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts)) * 60 * 60 * 1000000);
       v_ret_value := v_ret_value + ((EXTRACT(MINUTE FROM v_end_ts) - EXTRACT(MINUTE FROM v_start_ts)) * 60 * 1000000);
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(SECOND FROM v_end_ts) - EXTRACT(SECOND FROM v_start_ts)) * 1000000));
    ElSIF v_part IN ('NANOSECOND', 'NS') THEN
       v_ret_value := ROUND(v_end_date - v_start_date) * 24 * 60 * 60 * 1000000000;
       v_ret_value := v_ret_value + ((EXTRACT(HOUR FROM v_end_ts) - EXTRACT(HOUR FROM v_start_ts)) * 60 * 60 * 1000000000);
       v_ret_value := v_ret_value + ((EXTRACT(MINUTE FROM v_end_ts) - EXTRACT(MINUTE FROM v_start_ts)) * 60 * 1000000000);
       v_ret_value := ROUND(v_ret_value + ((EXTRACT(SECOND FROM v_end_ts) - EXTRACT(SECOND FROM v_start_ts)) * 1000000000));
    END IF;
    RETURN v_ret_value;
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datediff_sqlserver;

--http://st-doc.us.oracle.com/database/121/SQLRF/functions066.htm#SQLRF00639
--TIMESTAMP-TIMESTAMP = INTERVAL DAY TO SECOND
FUNCTION datediff_sybase(p_datepart IN VARCHAR2, p_start_date_expr IN TS, p_end_date_expr IN TS)
RETURN NUMBER
IS
    v_ret_value NUMBER := NULL;
    v_diff INTERVAL DAY(9) TO SECOND(9);
    v_years NUMBER;
    v_weeks NUMBER;
    v_days NUMBER;
    v_hours NUMBER;
    v_minutes NUMBER;
    v_seconds NUMBER;
    v_milliseconds NUMBER;
    v_microseconds NUMBER;
BEGIN
    v_diff := p_end_date_expr -  p_start_date_expr;
    --TIMESTAMP - TIMESTAMP  = INTERVAL DAY TO SECOND
    v_days := EXTRACT(DAY FROM v_diff);
    v_hours := EXTRACT(HOUR FROM v_diff);
    v_minutes :=EXTRACT(MINUTE FROM v_diff);
    v_seconds :=EXTRACT(SECOND FROM v_diff);
    --MILLISECONDS: In Sybase only 1 second precision is taken into account
    v_milliseconds := v_seconds - TRUNC(v_seconds);--full milliseconds (0.123456)
    v_milliseconds := ROUND(v_milliseconds*10)*100;--only 1 precision of milliseconds (100);
    v_microseconds := v_seconds - TRUNC(v_seconds);--full microseconds(0.123456)
    v_microseconds := TRUNC(v_microseconds * 1000000);
    
    IF p_datepart IN ('WEEK', 'WK', 'WW') THEN
           v_ret_value :=  (NEXT_DAY(TRUNC(p_end_date_expr),'Sunday') -  NEXT_DAY(TRUNC(p_start_date_expr),'Sunday'))/7 ;
    ELSIF p_datepart IN ('DAY', 'DD', 'D','DAYOFYEAR', 'DY', 'Y') THEN
        --CANT USER THE INTERVAL DIFF(v_difF), AS EVEN 2 SECONDS COULD SEPERATE A DAY (1 second before midnight, 1 second after midnight
        --INSTEAD FIRST REMOVE HOURS/MINS FROM INITIAL VALUES THEN FIND THE INTERVAL
         v_ret_value :=  TRUNC(p_end_date_expr) -  TRUNC(p_start_date_expr);
    ElSIF p_datepart IN ('HOUR', 'HH') THEN
          v_ret_value := (v_days*24)+v_hours;
    ElSIF p_datepart IN ('MINUTE', 'MI', 'N') THEN
          v_ret_value :=(v_days*24*60)+(v_hours*60)+v_minutes;
    ElSIF p_datepart IN ('SECOND', 'SS', 'S') THEN
        --SECONDS CAN HAVE MILLISECONDS AS WELL, SO NEED TO TRUNCATE THESE IN SYBASE
          v_ret_value := (v_days*24*60*60)+(v_hours*60*60)+(v_minutes*60)+TRUNC(v_seconds);
    ElSIF p_datepart IN ('MILLISECOND', 'MS') THEN
          v_ret_value := (v_days*24*60*60*1000)+(v_hours*60*60*1000)+(v_minutes*60*1000)+(TRUNC(v_seconds)*1000)+v_milliseconds;
    ElSIF p_datepart IN ('MICROSECOND', 'MCS') THEN 
          v_ret_value := (v_days*24*60*60*1000000)+(v_hours*60*60*1000000)+(v_minutes*60*1000000)+(TRUNC(v_seconds)*1000000)+v_microseconds;
    ElSIF p_datepart IN ('NANOSECOND', 'NS') THEN
        --NANOSECONDS: In Sybase only 1 second precision is taken into account
          v_ret_value := (v_days*24*60*60*1000000)+(v_hours*60*60*1000000)+(v_minutes*60*1000000)+(TRUNC(v_seconds)*1000000000)+(v_microseconds*100);
    ELSIF p_datepart IN ('QUARTER', 'QQ', 'Q') THEN
       v_ret_value :=   (TO_CHAR(p_end_date_expr,'YYYY')*4 +
                         TO_CHAR(p_end_date_expr,'Q'))
                        -
                        (TO_CHAR(p_start_date_expr,'YYYY')*4  +
                         TO_CHAR(p_start_date_expr,'Q'));
    ELSIF p_datepart IN ('WEEKDAY', 'DW', 'W') THEN --WEEKDAY IN SYBASE RETURNS THE NUMBER OF WEEKS!
      v_ret_value :=  TRUNC((TRUNC(p_end_date_expr) -  TRUNC(p_start_date_expr))/7);
    ELSE   
          v_ret_value := datediff_sqlserver(p_datepart,p_start_date_expr,p_end_date_expr);
    END IF;
    RETURN v_ret_value;
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datediff_sybase;

FUNCTION datediff_(p_datepart IN VARCHAR2, p_start_date_expr IN TS, p_end_date_expr IN TS)
RETURN NUMBER
AS BEGIN
 IF DATABASE_TYPE = SYBASE THEN
    RETURN datediff_sybase(UPPER(p_datepart),p_start_date_expr,p_end_date_expr);
 ELSE
    RETURN datediff_sqlserver(p_datepart,p_start_date_expr,p_end_date_expr);
 END IF;
END datediff_;

FUNCTION NUMBERTODATE (P_NUM NUMBER) RETURN DATE
IS BEGIN
  RETURN ZERODATETIME()+P_NUM;
END;

FUNCTION datediff(p_datepart IN VARCHAR2, p_start_date_str IN VARCHAR2, p_end_date_str IN VARCHAR2)
RETURN NUMBER
IS
    v_start_ts  TS;
    v_end_ts  TS;
BEGIN
  v_start_ts := CONVERT_STRING_TO_TIMESTAMP(TRIM(p_start_date_str));
  v_end_ts := CONVERT_STRING_TO_TIMESTAMP(TRIM(p_end_date_str));
   RETURN datediff_(p_datepart, v_start_ts, v_end_ts);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datediff;

FUNCTION datediff(p_datepart IN VARCHAR2, p_start_date_expr IN DATE, p_end_date_expr IN DATE)
RETURN NUMBER
IS
BEGIN
      RETURN datediff_(p_datepart, p_start_date_expr, p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datediff;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN DATE, P_END_DATE_EXPR IN NUMBER) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,P_START_DATE_EXPR,NUMBERTODATE(P_END_DATE_EXPR));
END;

FUNCTION datediff(p_datepart IN VARCHAR2, p_start_date_expr IN TS, p_end_date_expr IN TS)
RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart, p_start_date_expr, p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END datediff;


FUNCTION datediff(P_DATEPART IN VARCHAR2, P_START_DATE_STR IN VARCHAR2, P_END_DATE_EXPR IN DATE) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart,  CONVERT_STRING_TO_TIMESTAMP(TRIM(p_start_date_str)), p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_STR IN VARCHAR2, P_END_DATE_EXPR IN NUMBER) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,P_START_DATE_STR,NUMBERTODATE(P_END_DATE_EXPR));
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_STR IN VARCHAR2, P_END_DATE_EXPR IN TS) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart,  CONVERT_STRING_TO_TIMESTAMP(TRIM(p_start_date_str)), p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_STR IN VARCHAR2, P_END_DATE_EXPR IN TSTZ) RETURN NUMBER
IS BEGIN
   RETURN DATEDIFF(P_DATEPART,P_START_DATE_STR , CAST(P_END_DATE_EXPR AS TS)); --REMOVE TZ
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN DATE, P_END_DATE_STR IN VARCHAR2) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart, p_start_date_expr,  CONVERT_STRING_TO_TIMESTAMP(TRIM(p_end_date_str)));
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN DATE, P_END_DATE_EXPR IN TS) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart, p_start_date_expr, p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;
 
FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN DATE, P_END_DATE_EXPR IN TSTZ) RETURN NUMBER
IS
BEGIN
     RETURN datediff(p_datepart, p_start_date_expr, CAST(p_end_date_expr AS TS));
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN NUMBER, P_END_DATE_EXPR IN DATE) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,NUMBERTODATE(P_START_DATE_EXPR),P_END_DATE_EXPR);
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN NUMBER, P_END_DATE_EXPR IN NUMBER) RETURN NUMBER
IS BEGIN
RETURN DATEDIFF(P_DATEPART,NUMBERTODATE(P_START_DATE_EXPR),NUMBERTODATE(P_END_DATE_EXPR));
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN NUMBER, P_END_DATE_STR IN VARCHAR2) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,NUMBERTODATE(P_START_DATE_EXPR),P_END_DATE_STR);
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN NUMBER, P_END_DATE_EXPR IN TS) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,NUMBERTODATE(P_START_DATE_EXPR),P_END_DATE_EXPR);
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN NUMBER, P_END_DATE_EXPR IN TSTZ) RETURN NUMBER
IS BEGIN
  RETURN DATEDIFF(P_DATEPART,NUMBERTODATE(P_START_DATE_EXPR),P_END_DATE_EXPR);
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TS, P_END_DATE_EXPR IN DATE) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart, p_start_date_expr, p_end_date_expr);
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TS, P_END_DATE_EXPR IN NUMBER) RETURN NUMBER
IS BEGIN
 RETURN DATEDIFF(P_DATEPART,P_START_DATE_EXPR,NUMBERTODATE(P_END_DATE_EXPR));
END;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TS, P_END_DATE_STR IN VARCHAR2) RETURN NUMBER
IS
BEGIN
     RETURN datediff_(p_datepart, p_start_date_expr,  CONVERT_STRING_TO_TIMESTAMP(TRIM(p_end_date_str)));
EXCEPTION
     WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TS, P_END_DATE_EXPR IN TSTZ) RETURN NUMBER
IS BEGIN
     RETURN datediff(p_datepart, p_start_date_expr,  CAST(P_END_DATE_EXPR AS TS));--REMOVE TZ
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TSTZ, P_END_DATE_EXPR IN TSTZ) RETURN NUMBER
IS BEGIN
RETURN DATEDIFF(P_DATEPART,CAST(P_START_DATE_EXPR AS TS),P_END_DATE_EXPR);--REMOVE TZ
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TSTZ, P_END_DATE_EXPR IN TS) RETURN NUMBER
IS BEGIN
RETURN DATEDIFF(P_DATEPART,CAST(P_START_DATE_EXPR AS TS),P_END_DATE_EXPR);--REMOVE TZ
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TSTZ, P_END_DATE_EXPR IN DATE) RETURN NUMBER
IS BEGIN
RETURN DATEDIFF(P_DATEPART,CAST(P_START_DATE_EXPR AS TS),P_END_DATE_EXPR);--REMOVE TZ
END DATEDIFF;

FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TSTZ, P_END_DATE_EXPR IN NUMBER) RETURN NUMBER
IS BEGIN
 RETURN DATEDIFF(P_DATEPART,P_START_DATE_EXPR,NUMBERTODATE(P_END_DATE_EXPR));
END;
FUNCTION DATEDIFF(P_DATEPART IN VARCHAR2, P_START_DATE_EXPR IN TSTZ, P_END_DATE_STR IN VARCHAR2) RETURN NUMBER
IS BEGIN
RETURN DATEDIFF(P_DATEPART,CAST(P_START_DATE_EXPR AS TS),P_END_DATE_STR);--REMOVE TZ
END DATEDIFF;

FUNCTION month_(p_date_str IN VARCHAR2)
RETURN NUMBER
IS
    v_date DATE;
    v_dateformat VARCHAR2(50);
BEGIN
    v_date := CONVERT_STRING_TO_TIMESTAMP(p_date_str);
    IF v_date IS NULL THEN
      RETURN NULL;
    END IF;
    
    RETURN TO_NUMBER(TO_CHAR(v_date, 'MM'));
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END MONTH_;

FUNCTION month_(p_date_expr IN DATE)
RETURN NUMBER
IS
BEGIN
    RETURN EXTRACT(MONTH FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END MONTH_;

FUNCTION month_(p_date_expr IN TS)
RETURN NUMBER
IS
BEGIN
    RETURN EXTRACT(MONTH FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END MONTH_;

FUNCTION MONTH_(P_DATE_EXPR  IN TSTZ) RETURN NUMBER
IS
BEGIN
    RETURN MONTH_(CAST(p_date_expr AS TS));
END MONTH_;

-- PUBLIC NOT REFACTORED YET
FUNCTION rand(p_seed NUMBER DEFAULT NULL)
RETURN NUMBER
IS
    v_rand_num NUMBER;
BEGIN
      IF p_seed IS NOT NULL THEN
         DBMS_RANDOM.SEED(p_seed);
      END IF;
      
      v_rand_num := DBMS_RANDOM.VALUE();
      
      RETURN v_rand_num;
EXCEPTION
     WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END rand;

PROCEDURE resetTrancount
IS
BEGIN	
   trancount := 0;
END resetTrancount;

FUNCTION isnumeric(p_expr IN VARCHAR2)
RETURN NUMBER
IS
    numeric_val NUMBER;
    temp_str VARCHAR2(50);
BEGIN
    temp_str := p_expr;
    IF SUBSTR(temp_str, 1, 1) = '$' THEN
       temp_str := SUBSTR(temp_str, 2);
    END IF;
    
    numeric_val := TO_NUMBER(temp_str);
    RETURN 1;
EXCEPTION
    WHEN OTHERS THEN
       RETURN 0;
END isnumeric;

FUNCTION stats_date(p_table IN VARCHAR2, p_index IN VARCHAR2)
RETURN DATE
IS
    v_last_analyzed DATE;
BEGIN
    SELECT last_analyzed INTO v_last_analyzed
      FROM USER_IND_STATISTICS
     WHERE table_name LIKE UPPER(p_table)
       AND index_name LIKE UPPER(p_index);
  
    RETURN v_last_analyzed;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END stats_date;

FUNCTION oct(p_num VARCHAR2)
RETURN VARCHAR2
IS
BEGIN
    RETURN to_base(p_num, 8);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END oct;

FUNCTION quotename(p_str IN VARCHAR2, p_delimiters IN VARCHAR2 DEFAULT '[]')
RETURN VARCHAR2
IS
    v_ret_val VARCHAR2(150) := NULL;
BEGIN
    IF p_delimiters = '[]' THEN
       v_ret_val := '[' || REPLACE(p_str, ']', ']]') || ']';
    ELSIF p_delimiters = '"' THEN
       v_ret_val := '"' || p_str || '"';
    ELSIF p_delimiters = '''' THEN
       v_ret_val := '''' || p_str || '''';
      END IF;
     
      RETURN v_ret_val;
EXCEPTION
      WHEN OTHERS THEN
         raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END quotename;

FUNCTION parsename(p_object_name IN VARCHAR2, p_object_piece IN NUMBER)
RETURN VARCHAR2
IS
    ret_val VARCHAR2(150) := NULL;
    pos NUMBER;
    v_next_pos NUMBER;
BEGIN
    IF p_object_name IS NULL THEN 
       RETURN NULL;
    END IF;
    
    -- for 10g
    IF NOT DBMS_DB_VERSION.VER_LE_9_2 THEN
      IF p_object_piece = 1 THEN -- object name
         ret_val := REGEXP_SUBSTR(p_object_name, '(^[^\.]+$)|(\.[^\.]+$)');
         ret_val := REPLACE(ret_val, '.', '');
      ELSIF p_object_piece = 2 THEN -- schema name
         ret_val := REGEXP_SUBSTR(p_object_name, '([^\.]+)\.([^\.]+$)');
         ret_val := REGEXP_REPLACE(ret_val, '\.([^\.]+$)', '');
      ELSIF p_object_piece = 3 THEN -- database name
         ret_val := REGEXP_SUBSTR(p_object_name, '([^\.]+)\.([^\.]*)\.([^\.]+$)');
         ret_val := REGEXP_REPLACE(ret_val, '\.([^\.]*)\.([^\.]+$)', '');
      ELSIF p_object_piece = 4 THEN -- server name
         ret_val := REGEXP_SUBSTR(p_object_name, '^([^\.]+)\.([^\.]*)\.([^\.]*)\.([^\.]+$)');
         IF ret_val IS NOT NULL THEN
           ret_val := REGEXP_REPLACE(p_object_name, '^([^\.]+)\.([^\.]*)\.([^\.]*)\.([^\.]+$)', '\1');
         END IF;
      END IF;
    ELSE
      ret_val := p_object_name;
      v_next_pos := LENGTH(p_object_name);
      FOR i IN 1 .. p_object_piece LOOP
        pos := INSTR(p_object_name, '.', -1, i);
        IF pos > 0 THEN
          ret_val := SUBSTR(p_object_name, pos + 1, v_next_pos - pos);
        END IF;
        v_next_pos := pos;
      END LOOP;
      
      IF LENGTH(ret_val) = 0 THEN
        RETURN NULL;
      END IF;
    END IF;
    
    RETURN ret_val;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END PARSENAME;

FUNCTION fetch_status(p_cursorfound IN BOOLEAN)
RETURN NUMBER
IS
     v_fetch_status NUMBER := 0;
BEGIN
   CASE
     WHEN p_cursorfound THEN
        v_fetch_status := 0; --fetch successful
     ELSE
        v_fetch_status := -1; --fetch unsuccessful
     END CASE;
     RETURN v_fetch_status;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END fetch_status;

FUNCTION sqlstatus(p_cursorfound IN BOOLEAN,p_cursoropen IN BOOLEAN)
RETURN NUMBER
IS
     v_sqlerror NUMBER := 0;
BEGIN
   CASE
     WHEN NOT p_cursoropen THEN
        v_sqlerror := 1; -- fetch statement resulted in an error
     WHEN p_cursorfound THEN
        v_sqlerror := 0; --successful completion
     ELSE 
 		v_sqlerror :=2;  --no more data in result set
     END CASE;
     RETURN v_sqlerror;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END sqlstatus;

FUNCTION ident_seed(p_sequence IN VARCHAR2)
RETURN NUMBER
IS
    v_seed NUMBER;
BEGIN
      SELECT min_value INTO v_seed
         FROM USER_SEQUENCES
         WHERE sequence_name LIKE UPPER(p_sequence);
  
      RETURN v_seed;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END IDENT_SEED;

FUNCTION to_base(p_dec NUMBER, p_base NUMBER) 
RETURN VARCHAR2
IS
    v_str VARCHAR2(255);
    v_num NUMBER;
    v_hex VARCHAR2(16) DEFAULT '0123456789ABCDEF';
BEGIN
    v_num := p_dec;
    
    IF p_dec IS NULL OR p_base IS NULL THEN
      RETURN NULL;
    END IF;

    IF TRUNC(p_dec) <> p_dec OR p_dec < 0 THEN
        RAISE PROGRAM_ERROR;
    END IF;
    
    LOOP
      v_str := SUBSTR(v_hex, MOD(v_num, p_base) + 1, 1) || v_str;
      v_num := TRUNC(v_num / p_base);
      
      EXIT WHEN v_num = 0;
    END LOOP;
    
    RETURN v_str;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END TO_BASE;

FUNCTION TO_SECOND(P_TIME VARCHAR2) 
RETURN NUMBER
IS
  v_hours NUMBER;
  v_minutes NUMBER;
  v_seconds NUMBER;
  v_tot_seconds NUMBER;
BEGIN
  v_hours := REGEXP_REPLACE(P_TIME, '((\d{1,2})\:(\d{2})\:(\d{2}))', '\2');
  v_minutes := REGEXP_REPLACE(P_TIME, '((\d{1,2})\:(\d{2})\:(\d{2}))', '\3');
  v_seconds := REGEXP_REPLACE(P_TIME, '((\d{1,2})\:(\d{2})\:(\d{2}))', '\4');
  v_tot_seconds := v_hours * 60 * 60 + v_minutes * 60 + v_seconds;
  RETURN v_tot_seconds;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END TO_SECOND;

PROCEDURE decrementTrancount
IS
BEGIN	
	IF trancount > 0 THEN
      	trancount := trancount - 1;
   	END IF;
END decrementTrancount;

FUNCTION hex(p_num VARCHAR2)
RETURN VARCHAR2
IS
BEGIN
    RETURN to_base(p_num, 16);
  EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END hex;

FUNCTION isdate(p_expr IN VARCHAR2)
RETURN NUMBER
IS
     v_is_valid_date BINARY_INTEGER := 0;
BEGIN
    IF CONVERT_STRING_TO_TIMESTAMP(p_expr) IS NOT NULL THEN
       RETURN 1;
    ELSE 
       SELECT NVL2(TO_DATE(p_expr), 1, 0) INTO v_is_valid_date FROM DUAL;
       RETURN v_is_valid_date;
    END IF;    
EXCEPTION 
    WHEN OTHERS THEN
       RETURN 0;
END isdate;

FUNCTION radians(p_degree IN NUMBER)
RETURN NUMBER
IS
    v_rad NUMBER;
BEGIN
    v_rad := p_degree / 180 * pi();
    
    IF INSTR(TO_CHAR(p_degree),'.') = 0 THEN
        v_rad := FLOOR(v_rad);
    ELSE 
        v_rad := round(p_degree / 180 * pi(),18); 
    END IF;
      
    RETURN v_Rad ;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END radians;

FUNCTION GETIDENTITY RETURN NUMBER
IS
BEGIN
 RETURN IDENTITY_VALUE;
END;

FUNCTION ident_incr(p_sequence IN VARCHAR2)
RETURN NUMBER
IS
    v_incr_by NUMBER;
BEGIN
    SELECT increment_by INTO v_incr_by
       FROM USER_SEQUENCES
       WHERE sequence_name LIKE UPPER(p_sequence);

    RETURN v_incr_by;
EXCEPTION
    WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ident_incr;

FUNCTION day_(p_date_str IN VARCHAR2)
RETURN NUMBER
IS
    v_date DATE;
BEGIN
    v_date := CONVERT_STRING_TO_TIMESTAMP(p_date_str);
    IF v_date IS NULL THEN
      RETURN NULL;
    END IF;
    
    RETURN TO_NUMBER(TO_CHAR(v_date, 'DD'));
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_;

FUNCTION day_(p_date_expr IN DATE)
RETURN NUMBER
IS
BEGIN
    RETURN EXTRACT(DAY FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_;

FUNCTION day_(p_date_expr IN TS)
RETURN NUMBER
IS
BEGIN
    RETURN EXTRACT(DAY FROM p_date_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_;

FUNCTION DAY_(P_DATE_EXPR    IN TSTZ) RETURN NUMBER
IS
BEGIN
    RETURN DAY_(CAST(p_date_expr AS TS));
END DAY_;

FUNCTION stuff(p_expr VARCHAR2, p_startIdx NUMBER, p_len NUMBER, p_replace_expr VARCHAR2) 
RETURN VARCHAR2
IS
BEGIN
       RETURN REPLACE(p_expr, SUBSTR(p_expr, p_startIdx, p_len), p_replace_expr);
EXCEPTION
        WHEN OTHERS THEN
          raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END stuff;
PROCEDURE commit_transaction
IS
BEGIN	
   IF trancount <= 1 THEN
        COMMIT;
   END IF;
   resetTrancount;
END commit_transaction;

--RETURN A NUMBER TO HELP REPLICATE @@TRANSTATE
FUNCTION  COMMIT_TRANSACTION_STATE RETURN NUMBER
IS
BEGIN	
   IF trancount <= 1 THEN
        COMMIT;
   END IF;
   resetTrancount;
   RETURN 1;
EXCEPTION WHEN OTHERS THEN
  RETURN 3;   
END COMMIT_TRANSACTION_STATE;

FUNCTION round_(p_expr NUMBER, p_len NUMBER, p_function NUMBER DEFAULT 0) 
RETURN NUMBER
IS
    v_ret_value NUMBER;
BEGIN
      IF p_function = 0 THEN
         v_ret_value := ROUND(p_expr, p_len);
      ELSE
         v_ret_value := TRUNC(p_expr, p_len);
      END IF;
      
      RETURN v_ret_value;
EXCEPTION
     WHEN OTHERS THEN
       raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END round_;

FUNCTION patindex(p_pattern IN VARCHAR2, p_expr IN VARCHAR2, p_format IN VARCHAR2)
RETURN NUMBER
IS
    v_search_pattern VARCHAR2(100);
    v_pos NUMBER := 0;
    v_charsfmt VARCHAR2(20) := 'using chars';
    v_charactersfmt  VARCHAR2(20) := 'using characters';
    v_bytesfmt VARCHAR2(20) := 'using bytes';
    v_format VARCHAR2(20);
    v_errmsg VARCHAR2(50) := 'Invalid format: ';
BEGIN
      IF p_pattern IS NULL OR p_expr IS NULL THEN
         RETURN NULL;
      END IF;
      
      IF NOT DBMS_DB_VERSION.VER_LE_9_2 THEN
        v_search_pattern := p_pattern;
        v_search_pattern := REPLACE(v_search_pattern, '\', '\\');
        v_search_pattern := REPLACE(v_search_pattern, '*', '\*');
        v_search_pattern := REPLACE(v_search_pattern, '+', '\+');
        v_search_pattern := REPLACE(v_search_pattern, '?', '\?');
        v_search_pattern := REPLACE(v_search_pattern, '|', '\|');
        v_search_pattern := REPLACE(v_search_pattern, '^', '\^');
        v_search_pattern := REPLACE(v_search_pattern, '$', '\$');
        v_search_pattern := REPLACE(v_search_pattern, '.', '\.');
        v_search_pattern := REPLACE(v_search_pattern, '{', '\{');
        v_search_pattern := REPLACE(v_search_pattern, '_', '.');
              
        v_format := lower(p_format);
        IF v_format = v_charsfmt OR v_format = v_charactersfmt THEN
           IF SUBSTR(v_search_pattern, 1, 1) != '%' AND 
              SUBSTR(v_search_pattern, -1, 1) != '%' THEN
               v_search_pattern := '^' || v_search_pattern || '$';
           ELSIF SUBSTR(v_search_pattern, 1, 1) != '%' THEN
               v_search_pattern := '^' || SUBSTR(v_search_pattern, 1, LENGTH(v_search_pattern) - 1);
           ELSIF SUBSTR(v_search_pattern, -1, 1) != '%' THEN
               v_search_pattern := SUBSTR(v_search_pattern, 2) || '$';
           ELSE
               v_search_pattern := SUBSTR(v_search_pattern, 2, LENGTH(v_search_pattern) - 2);
           END IF;    
        ELSIF v_format = v_bytesfmt THEN    
           IF SUBSTRB(v_search_pattern, 1, 1) != '%' AND 
              SUBSTRB(v_search_pattern, -1, 1) != '%' THEN
               v_search_pattern := '^' || v_search_pattern || '$';
           ELSIF SUBSTRB(v_search_pattern, 1, 1) != '%' THEN
               v_search_pattern := '^' || SUBSTRB(v_search_pattern, 1, LENGTHB(v_search_pattern) - 1);
           ELSIF SUBSTRB(v_search_pattern, -1, 1) != '%' THEN
               v_search_pattern := SUBSTRB(v_search_pattern, 2) || '$';
           ELSE
               v_search_pattern := SUBSTRB(v_search_pattern, 2, LENGTHB(v_search_pattern) - 2);
           END IF;    
        ELSE
            v_errmsg := v_errmsg || p_format;
            raise_application_error(-20001, v_errmsg);
        END IF;
        v_pos := REGEXP_INSTR(p_expr, v_search_pattern);
      ELSE 
        v_pos := 0;
      END IF;
      
      RETURN v_pos;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END patindex;
PROCEDURE incrementTrancount
IS
BEGIN	
   trancount := trancount + 1;
END INCREMENTTRANCOUNT;

FUNCTION reverse_(p_expr IN VARCHAR2)
RETURN VARCHAR2
IS
    v_result VARCHAR2(2000) := NULL;
BEGIN      
    FOR i IN 1..LENGTH(p_expr) LOOP
      v_result := v_result || SUBSTR(p_expr, -i, 1);
    END LOOP;
  
    RETURN v_result;    
EXCEPTION 
    WHEN OTHERS THEN
      raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END reverse_;

PROCEDURE SET_DATABASE_TYPE (P_DATABASE_TYPE IN VARCHAR2)
AS
BEGIN
DATABASE_TYPE := P_DATABASE_TYPE;
END SET_DATABASE_TYPE;

FUNCTION SPID  RETURN NUMBER
AS 
BEGIN
RETURN USERENV('sessionid');
END;

FUNCTION STR(P_EXPR IN NUMBER, P_LEN IN NUMBER DEFAULT 10, P_SCALE IN NUMBER DEFAULT 0)  RETURN VARCHAR2
IS
    v_ret_val VARCHAR2(50);
    v_temp_val NUMBER;
    v_format_str VARCHAR2(50);
    v_len NUMBER;
    v_val NUMBER;
    v_has_decimal BOOLEAN := FALSE;
BEGIN
      IF INSTR(TO_CHAR(p_expr), '.') > 0 THEN
        v_has_decimal := TRUE;
        v_len := LENGTH(SUBSTR(TO_CHAR(p_expr), 1, INSTR(TO_CHAR(p_expr), '.')-1));
      ELSE
        v_len := LENGTH(TO_CHAR(p_expr));
      END IF;  
      
      IF p_len < v_len THEN
         RETURN TRIM(LPAD(' ', p_len+1,'*'));
      END IF;
      
      v_temp_val := p_expr;
      IF p_len >= v_len THEN
         v_temp_val := ROUND(v_temp_val, p_scale);
      ELSE
         v_temp_val := ROUND(v_temp_val, 0);
      END IF;   
      
      IF p_scale > 0 AND v_has_decimal THEN
         IF v_len >= (p_len - p_scale) THEN
             v_format_str := LPAD(' ', v_len+1, '9');
         ELSE    
             v_format_str := LPAD(' ', (p_len - p_scale), '9');
         END IF;
         v_format_str := TRIM(v_format_str);
         IF INSTR(TO_CHAR(p_expr), '.') != p_len THEN 
           v_format_str := v_format_str || '.';
           v_format_str := RPAD(v_format_str, p_len, '9');
         END IF;  
      ELSE
         v_format_str := TRIM(LPAD(' ', p_len+1, '9'));
      END IF;
      v_ret_val := TO_CHAR(v_temp_val, v_format_str);
      RETURN v_ret_val;
EXCEPTION 
      WHEN OTHERS THEN
        raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END STR;

PROCEDURE IDENTITY_RESET( V_IDENTITYID VARCHAR2) AS
BEGIN
identitymap(v_identityid) := 0;
END;

FUNCTION IDENTITY( V_IDENTITYID VARCHAR2, V_SEED  INT DEFAULT 1, V_INCREMENT  INT DEFAULT 1) RETURN NUMBER
as
  v_current number(20) := 0;
begin
  dbms_output.put_line('v_current1:'||v_current);
  begin
  v_current := identitymap(v_identityid);
  exception when others then
   v_current := 0;
  END;
  dbms_output.put_line('v_current2:'||v_Current);
  if v_current = 0 then
    identitymap(v_identityID):= 0;
  end if;
  v_current := identitymap(v_identityid);
  v_current := v_current + 1;
  identitymap(v_identityid) := v_current;
  return v_current;
end;

FUNCTION OBJECT_ID(objectref VARCHAR2) RETURN INT
AS
  ownerName  VARCHAR2(128);
  objectName VARCHAR2(128);
  objectId   INT;
BEGIN
  --select * from all_objects;
  --select object_id('system.redo_log') from dual;
  IF( INSTR(objectref,'.') = 0 )THEN
    ownerName             := NULL;
    objectName            := objectref;
  ELSE
    ownerName  := SUBSTR(objectref,0,INSTR(objectref,'.')-1);
    objectName := SUBSTR(objectref,INSTR(objectref,'.')  +1);
  END IF;
  BEGIN
    SELECT object_id
    INTO objectId
    FROM all_objects
    WHERE UPPER(owner)     = UPPER(NVL(ownerName,owner))
    AND  UPPER(object_name) = UPPER(objectName) ;
  EXCEPTION WHEN OTHERS THEN objectId := NULL;
  END;
  RETURN objectId;
END;

PROCEDURE RAISERROR(ERRORCODE NUMBER,MSG VARCHAR2)
AS
BEGIN
    -- NOTE: Oracle raise_application_error will terminate normal code flow , which T-SQL raiserror does not.
	-- raise_application_error(ERRORCODE||':'||MSG);
	DBMS_OUTPUT.PUT_LINE(ERRORCODE||':'||MSG);
END;

PROCEDURE HANDLEERROR(ERRORCODE NUMBER,MSG VARCHAR2)
AS
BEGIN
    -- NOTE: Oracle raise_application_error will terminate normal code flow , which T-SQL raiserror does not.
    raise_application_error(-20002,ERRORCODE||':'||MSG);
	--DBMS_OUTPUT.PUT_LINE(ERRORCODE||':'||MSG);
END;

FUNCTION error_line RETURN VARCHAR2
AS
BEGIN
 RETURN '0';
END;

FUNCTION error_procedure RETURN VARCHAR2
AS
BEGIN
 RETURN '0';
END;

FUNCTION error_severity RETURN VARCHAR2
AS
BEGIN
 RETURN '0';
END;

FUNCTION error_state RETURN VARCHAR2
AS
BEGIN
 RETURN '0';
END;

BEGIN

-- Datetime formats/styles starting with DAY only --
DT_DAY := VARCHAR2_ARRAY(
----------- SQLServer and Sybase Datetime formats/styles --------------
----- Format/Style ------       ------ Standard ------ 
'fxfmdd/mm/yy',                    --British/French
'fxfmdd/mm/yyyy',                  --British/French(with Century)
'fxfmdd.mm.yy',                    --German(without Century)
'fxfmdd.mm.yyyy',                  --German(with Century)
'fxfmdd-mm-yy',                    --Italian(without Century)
'fxfmdd-mm-yyyy',                  --Italian(with Century)
'fxfmdd mon yy',                   --(without Century)
'fxfmdd mon yyyy',                 --(with Century)
'dd mon yyyy hh24:mi:ssxff3',      --Europe default+milliseconds
'fxdd mon yyyy hh12:mi:ss:ff3AM',  --Hijri calendar system 
'fxdd/mm/yy hh12:mi:ss:ff3AM',     --Hijri calendar system 

-------- Some more DateTime formats that were addressed before --------
'fmdd yyyy MONTH',

----------------- Additional DateTime formats ------------------------- 
'fmdd Month',
'fmdd Month yy',
'fmdd Month yyyy',
'fxddmmyy',
'fxddmmyyyy',
'fxdd-Mon-yy',
'fxdd-MON-yy',
'fxdd-Mon-yyyy',
'fxdd-MON-yyyy'
);

-- Datetime formats/styles starting with MONTH only --
DT_MONTH := VARCHAR2_ARRAY(
----------- SQLServer and Sybase Datetime formats/styles --------------
----- Format/Style ------       ------ Standard ------ 
'fxfmmm/dd/yyyy hh24:mi:ss',       --SQLSERVER 101  
'fmmon dd yyyy hh:miAM',           --Default   
'fxfmmm/dd/yy',                    --US(without Century)
'fxfmmm/dd/yyyy',                  --US(with Century)
'fxfmMon dd, yy',                  --(without Century)
'fxfmMon dd, yyyy',                --(with Century)
'mon dd yyyy hh12:mi:ssxff3am',    --Default+milliseconds
'fxfmmm-dd-yy',                    --US(without Century)
'fxfmmm-dd-yyyy',                  --US(with Century)

-------- Some more DateTime formats that were addressed before -------- 
'fmMONTH, yyyy',
'MON yyyy',
'fmMONTH dd, yyyy',
----------------- Additional DateTime formats ------------------------- 
'mm/yy',
'mm/yyyy',
'fmMonth dd, yyyy',             
'fmMonth dd',
'mm-yy',
'mm-yyyy',
'fxmmddyy',
'fxmmddyyyy',
'Mon-yy',
'Mon-yyyy',
'MON-yy',
'MON-yyyy'
);

-- Datetime formats/styles starting with YEAR only --
DT_YEAR := VARCHAR2_ARRAY(
----------- SQLServer and Sybase Datetime formats/styles --------------
----- Format/Style ------       ------ Standard ------ 
'fxfmyy.mm.dd',                    --ANSI(without Century)
'fxfmyyyy.mm.dd',                  --ANSI(with Century)
'fxfmyy/mm/dd',                    --Japan(without Century)
'fxfmyyyy/mm/dd',                  --Japan(with Century)
'yymmdd',                          --ISO(without Century)
'yyyymmdd',                        --ISO(with Century)
'fxyyyy-mm-dd hh24:mi:ss',         --ODBC canonical
'fxyyyy-mm-dd hh24:mi:ssxff3',     --ODBC canonical (with milliseconds)
'fxyyyy-mm-dd"T"hh12:mi:ssxff3',   --ISO8601 (no spaces)
'fxyyyy-mm-dd hh12:mi:ssxff3',     --ISO8601 (with space)

-------- Some more DateTime formats that were addressed before -------- 
'fxyyyymmdd hh24:mi:ss',
'fxyyyy-mm-dd',                    --ISO8601 Date only
'fxyyyy-mm-dd hh12:mi:ss',
'yyyy MON',

----------------- Additional DateTime formats ------------------------- 
'yy/mm',
'yyyy/mm',
'yy-mm',
'yyyy-mm'
);

-- TIME formats/styles --
DT_TIME := VARCHAR2_ARRAY(
----------- SQLServer and Sybase Datetime formats/styles --------------
----- Format/Style ------       ------ Standard ------ 
'hh:mi:ss',
'hh24:mi:ssxff3',

-------- Some more Time formats that were addressed before ------------ 
'hh12 AM',                         --Time using Hour
'hh12:mi:ss AM'                    --Time
);

-- ORACLE Server NLS DATETIME FORMATS --
DT_NLS := VARCHAR2_ARRAY(
getNLSDATEformat,
getNLSTSformat,
getNLSTSformat
);

-- TIME formats/styles --
DT_TIMESTAMP := VARCHAR2_ARRAY(
'fxfmmm/dd/yyyy hh24:mi:ss.ff9',
'yyyy-mm-dd hh24:mi:ss.ff9'
);
/*
 You can modify the DT_FORMATS as you wish.
 Put the most likely Datetime Formats or 
 Nested Table Type variables at the top.
 For e.g: 
   DT_FORMATS := DT_MONTH MULTISET UNION DT_NLS;
   DT_FORMATS := DT_YEAR MULTISET UNION DT_TIME;
   DT_FORMATS := VARCHAR2_ARRAY('mm-dd-yy');
*/
DT_FORMATS := DT_MONTH MULTISET UNION DT_YEAR MULTISET UNION DT_NLS MULTISET UNION DT_TIMESTAMP;

END utils;