package javax.ide.progress;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;


/**
 * <code>ProgressManager</code> is a generic progress reporting system.
 *
 * @author <a href="mailto:chris.bensen@oracle.com">Chris Bensen</a>
 */

public class ProgressManager
{  
  private static HashMap<String, List<IProgressListener>> _listeners = null;
  private static Object _listenersMonitor = new Object();

  private ProgressManager()
  {
  }
  
  /**
   * beginProgress initiates the progress reporting process and performs initialization. Called before starting
   *  a process that will use the ProgressManager.
   */
  public static void beginProgress()
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners == null)
      {
        _listeners = new HashMap<String, List<IProgressListener>>();
      }
      else if (!_listeners.isEmpty())
      {
        _listeners.clear();
      }
    }
  }
  
  /**
   * endProgress ends the progress reporting process and performs cleanup. Called when done with the
   * ProgressManager.
   */
  public static void endProgress()
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        _listeners.clear();
        _listeners = null;
      }
    }
  }

  /**
   * registerListener registers a listener to a progress event. Because ProgressManager is a multi-cast
   * dispatch mechanism multiple listeners can listen to the events. 
   * @param name
   * @param listener
   */
  public static void registerListener(String name, IProgressListener listener)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> items = _listeners.get(name);
        
        if (items == null)
        {
          items = new LinkedList<IProgressListener>();
          items.add(listener);
          _listeners.put(name, items);
        }
        else
        {
          if (!items.contains(listener))
          {
            items.add(listener);
          }
        }
      }
    }
  }

  /**
   * removeListener removes a listener. This call is superfluous and it is not mandated that it be called unless
   * a specific listener need be removinged mid reporting. endProgress will perform any cleanup needed and
   * removing all listeners remaining.
   * @param name
   * @param listener
   */
  public static void removeListener(String name, IProgressListener listener)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> items = _listeners.get(name);
        
        if (items != null)
        {
          items.remove(listener);
        }
      }
    }
  }
  
  /**
   * completeProgress broadcasts the completeProgress event to all progress listeners.
   * @param name
   */
  public static void completeProgress(String name)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> listeners = _listeners.get(name);
        
        if (listeners != null)
        {
          for (IProgressListener listener : listeners)
          {
            if (listener != null)
            {
              listener.completeProgress();
            }
          }
        }
      }
    }
  }

  /**
   * updateProgressTitle broadcasts the updateProgressTitle event to all progress listeners.
   * @param name
   * @param title
   */
  public static void updateProgressTitle(String name, String title)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> listeners = _listeners.get(name);
        
        if (listeners != null)
        {
          for (IProgressListener listener : listeners)
          {
            if (listener != null)
            {
              listener.updateProgressTitle(title);
            }
          }
        }
      }
    }
  }
  
  /**
   * updateProgressDescription broadcasts the updateProgressDescription event to all progress listeners.
   * @param name
   * @param description
   * @param count
   */
  public static void updateProgressDescription(String name, String description, int count)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> listeners = _listeners.get(name);
        
        if (listeners != null)
        {
          for (IProgressListener listener : listeners)
          {
            if (listener != null)
            {
              listener.updateProgressDescription(description, count);
            }
          }
        }
      }
    }
  }
  
  /**
   * updateProgress broadcasts the updateProgress event to all progress listeners.
   * @param name
   * @param count
   */
  public static void updateProgress(String name, int count)
  {
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> listeners = _listeners.get(name);
        
        if (listeners != null)
        {
          for (IProgressListener listener : listeners)
          {
            if (listener != null)
            {
              listener.updateProgress(count);
            }
          }
        }
      }
    }
  }

  /**
   * Call finalizeProgress before calling updateProgress or updateProgressDescription. This will bake the number of
   * items that you want to update for this progres chunk.
   * @param name specifies the name of the progress chunk.
   * @param count is the number of items to update.
   */
  public static void finalizeProgress(String name, int count)
  {    
    synchronized (_listenersMonitor)
    {
      if (_listeners != null)
      {
        List<IProgressListener> listeners = _listeners.get(name);
        
        if (listeners != null)
        {
          for (IProgressListener listener : listeners)
          {
            if (listener != null)
            {
              listener.finalizeProgress(count);
            }
          }
        }
      }
    }
  }
}