/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.model;

import java.net.URI;
import java.util.Collection;

import javax.ide.spi.ProviderNotFoundException;
import javax.ide.Service;

/**
 *  The {@link DocumentFactory} interface is responsible for creating
 *  intances of {@link Document}s.<P>
 *
 *  The type of {@link Document} that gets created depends on the
 *  {@link URI} that is passed into the {@link #findOrCreate(URI)} method.  
 *  The {@link DocumentFactory}
 *  makes use of registered {@link Recognizer} instances to determine
 *  what {@link Document} class corresponds to a particular {@link URI}.<P>
 *
 *  JSR-198 specifies a number of ways to recognize documents:<p>
 *
 *  <ul>
 *    <li>1. By recognizing characteristics of the <code>URI</code>,<li><p>
 *    <li>2. For XML documents by recognizing certain markups, and <li><p>
 *    <li>3. By allowing extension writers to have full control of how a 
 *    document should be recognized.<li><p>
 *  </ul>
 *
 *  For case 1. extension writers need only map a file extension or some
 *  part of the <code>URI</code> to a document interface.<p>
 *
 *  For case 2. extension writers can map the XML namespace, doctype, or
 *  root element to a document interface.<p>
 *
 *  For case3. extension writers can introduce their own custom 
 *  <code>Recognizer</code>.<p>
 *
 *  Every {@link Document} instance created by the {@link DocumentFactory}
 *  is cached.  An instance of an already created {@link Document} can be
 *  retrieved from the cache by calling the {@link #find(URI)} method.
 *
 *  @see Document
 */
public abstract class DocumentFactory extends Service
{
  /**
   *  Returns the {@link Document} associated with the {@link URI}.  If the
   *  {@link Document} does not exist and the URI corresponds to a resource
   *  that <b>does</b> exists, a new {@link Document} of the correct 
   *  type is created.<P>
   *
   *  @return An existing document or a newly created one. Will return null if
   *      the specified uri does not exist.
   *
   *  @param uri unique {@link URI} identifying the document. Must not be null.
   *
   *  @exception  IllegalAccessException When the {@link Document} interface or
   *  its initializer is not accessible.
   *
   *  @exception  InstantiationException When the document is an abstract 
   *  class, an interface, an array class, a primitive type,
   *  or void; or if the instantiation fails for some other reason.
   *
   *  @exception ClassNotFoundException When the {@link Class} is not found.
   */
  public abstract Document findOrCreate( URI uri )
                  throws IllegalAccessException, 
                         InstantiationException,
                         ClassNotFoundException;


  /**
   *  Find the {@link Document} associated with the {@link URI}.  If the
   *  {@link Document} does not exist, <CODE>null</CODE> is returned.
   *
   *  @return  An existing {@link Document}, or <CODE>null</CODE> if none
   *  exists.
   *
   *  @param  uri  {@link URI} identifying the {@link Document}. Must not be
   *      null.
   */
  public abstract Document find( URI uri );

  /**
   *  Removes the <CODE>oldURI</CODE> from the cache and puts the
   *  <CODE>newURI</CODE> in the cache so that it is associated
   *  with the original {@link Document} <CODE>oldURI</CODE> pointed at.
   *
   *  @param oldURI the old URI of the document. Must not be null.
   *  @param newURI the new URI of the document. Must not be null.
   *  @return the cached document. May be null if no previous
   *      document was cached at this URI.
   */
  public abstract Document recache( URI oldURI, URI newURI );

  /**
   * Remove the specified URI from the cache. 
   * 
   * @param uri the uri to remove from the cache. Must not be null.
   * @return the document that was uncached, or null if no document was
   *    uncached.
   */
  public abstract Document uncache( URI uri );
  

  /**
   *  Returns a {@link Collection} of the {@link Document} instances that
   *  are currently cached.  The iteration order of the returned collection is
   *  not guaranteed.
   *  
   *  @return a collection of {@link Document}s. Never returns null, may return
   *      an empty collection. The collection is immutable.
   */
  public abstract Collection getCachedDocuments();


  /**
   * Get the DocumentFactory implementation for this IDE.
   * 
   * @return the DocumentFactory implementation for this IDE.
   */
  public static DocumentFactory getDocumentFactory()
  {
    try
    {
      return (DocumentFactory) getService( DocumentFactory.class );
    }
    catch ( ProviderNotFoundException lnfe )
    {
      lnfe.printStackTrace();
      throw new IllegalStateException( "No document factory." );
    }
  }
}

