/*
 * Copyright (c) 2005, 2014, Oracle and/or its affiliates.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

/**
 * A map that provides support for scoping.
 */
public final class ScopedMap extends AbstractMap
{
  private final Stack _mapStack = new Stack();

  public ScopedMap()
  {
    enterScope(); // Always have a root scope to avoid illegal object states
  }

  synchronized void reset()
  {
    while ( _mapStack.size() > 1 )
    {
      _mapStack.pop();
    }
  }

  /**
   * Enter a new scope level.
   */
  public synchronized void enterScope()
  {
    _mapStack.push( new LinkedHashMap( 8 ) );
  }

  /**
   * Exit the current scope level. Name / value pairs that
   * were defined in the current scope will no longer be available.
   */
  public synchronized void exitScope()
  {
    _mapStack.pop();
  }

  public synchronized Object put( Object key, Object value )
  {
    Map topMap = ( Map ) _mapStack.peek();
    return topMap.put( key, value );
  }

  public synchronized Object get( Object key )
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = ( Map ) i.next();
      Object o = thisMap.get( key );
      if ( o != null )
      {
        return o;
      }
    }
    return null;
  }

  public synchronized Object remove( Object key )
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = ( Map ) i.next();
      Object o = thisMap.remove( key );
      if ( o != null )
      {
        return o;
      }
    }
    return null;
  }

  public void clear()
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      ( ( Map ) i.next() ).clear();
    }
  }

  public synchronized Set entrySet()
  {
    HashSet hs = new HashSet();
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = ( Map ) i.next();
      hs.addAll( thisMap.entrySet() );
    }
    return hs;
  }

  public synchronized void copyInto( Map destination )
  {
    for ( Iterator i = _mapStack.reverseIterator(); i.hasNext(); )
    {
      Map thisMap = ( Map ) i.next();
      destination.putAll( thisMap );
    }
  }

  public static Map copyScopeData( Map scopeData, Set<String> keysToExclude )
  {
    HashMap destination = new HashMap( 7 );
    if ( scopeData instanceof ScopedMap )
    {
      ( ( ScopedMap ) scopeData ).copyInto( destination );
    }
    else
    {
      destination.putAll( scopeData );
    }

    if ( keysToExclude != null )
    {
      for ( String key : keysToExclude )
      {
        destination.remove( key );
      }
    }

    return destination;
  }

}
