/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.command;

import java.util.HashMap;
import javax.ide.model.Document;
import javax.ide.model.Element;
import javax.ide.model.Project;

/**
 * The Context class provides information about the state of the IDE when
 * a command is in the process of being executed. Such information 
 * includes:<p>
 * 
 * <ul>
 *    <li>the {@link Document} the user is working on,</li><p>
 *    <li>the {@link Project} the user is working on, and</li><p>
 *    <li>the currently selected {@link Element}s in the active 
 *    {@link javax.ide.view.View}.</li><p>
 * </ul>
 *
 * In addition, to this information, the context allows clients to store
 * arbitrary data to pass on to a command. Such data is stored in the 
 * context under a client specified string key. The data can be retrieved 
 * using that same key.
 */
public final class Context 
{
  private HashMap _properties = new HashMap();
  private Document _document;
  private Project _project;
  private Element[] _selection = new Element[ 0 ];

  /**
   * Constructor.
   * @param document The document the user is working on when a command is 
   * executed.
   * @param project The project the document is in.
   * @param selection The selected objects.
   */
  public Context( Document document, 
                  Project project,
                  Element[] selection )
  {
    _document = document;
    _project = project;
    _selection = selection;
  }

  /**
   * Constructor.
   * @param document The document the user is working on when a command is 
   * executed.
   * @param project The project the document is in.
   */
  public Context( Document document, 
                  Project project)
  {
    this( document, project, null );
  }

  /**
   * The current {@link Project} the user is working on.
   */
  public Project getProject()
  {
    return _project;
  }

  /**
   * The current {@link Document} the user is working on.
   */
  public Document getDocument()
  {
    return _document;
  }

  /**
   * Get the currently selected objects. These are {@link Element}s
   * that users can select inside IDE views.
   *
   * @return the currently selected objects. If nothing selected this method 
   * returns an empty array.
   */
  public Element[] getSelection()
  {
    return _selection;
  }
  
  /**
   * Set the currently selected objects. These are {@link Element}s 
   * that users can select inside IDE views.
   *
   * @param selection An array of {@link Element}s.
   */
  public void setSelection( Element[] selection )
  {
    _selection = selection != null ? selection : new Element[ 0 ];
  }

  /**
   * Retrieves the value associated with a property.  If no value
   * exists for the requested property, the specified default value
   * is returned. Use context properties to pass along extra data 
   * meaningful to an extension.
   *
   * @param name the property key for which a value is desired.
   * @param defaultValue the value to return if no value currently
   * exists.
   * 
   * @return the value of the requested property, or the default value
   * if the property does not exist.
   */
  public Object getProperty( String name, Object defaultValue )
  {
    final Object  property = _properties.get( name );

    return ( property == null ? defaultValue : property );
  }

  /**
   * Sets the value for a property. Setting a value to <CODE>null</CODE>
   * removes that property.  Use context properties to pass along extra data 
   * meaningful to an extension.
   * 
   * @param name the property key to set
   * @param value the value to set
   */
  public void setProperty( String name, Object value )
  {
    if ( name != null )
    {
      if ( value == null )
      {
        _properties.remove( name );
      }
      else
      {
        _properties.put( name, value );
      }
    }
  }
}
