/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.security.login;

import java.rmi.NotBoundException;
import java.rmi.RemoteException;
import java.util.List;
import java.util.logging.Logger;
import javax.security.auth.Subject;
import oracle.kv.impl.security.SessionAccessException;
import oracle.kv.impl.security.login.LoginManager;
import oracle.kv.impl.security.login.LoginToken;
import oracle.kv.impl.security.login.SessionId;
import oracle.kv.impl.security.login.TokenResolver;
import oracle.kv.impl.security.login.TopologyResolver;
import oracle.kv.impl.security.login.TrustedLoginAPI;
import oracle.kv.impl.security.login.UserLoginAPI;
import oracle.kv.impl.topo.AdminId;
import oracle.kv.impl.topo.RepNodeId;
import oracle.kv.impl.topo.ResourceId;
import oracle.kv.impl.topo.StorageNodeId;
import oracle.kv.impl.util.registry.RegistryUtils;

public class TokenResolverImpl
implements TokenResolver {
    private static final int MAX_RESOLVE_RNS = 10;
    private static final int RESOLVE_FAIL_LIMIT = 2;
    private final String hostname;
    private final int registryPort;
    private volatile String storeName;
    private final TopologyResolver topoResolver;
    private volatile TokenResolver persistentResolver;
    private final LoginManager loginMgr;
    private volatile Logger logger;

    public TokenResolverImpl(String hostname, int registryPort, String storeName, TopologyResolver topoResolver, LoginManager internalLoginMgr, Logger logger) {
        this.hostname = hostname;
        this.registryPort = registryPort;
        this.storeName = storeName;
        this.topoResolver = topoResolver;
        this.loginMgr = internalLoginMgr;
        this.persistentResolver = null;
        this.logger = logger;
    }

    public void setLogger(Logger newLogger) {
        this.logger = newLogger;
    }

    public void setStoreName(String newStoreName) {
        this.storeName = newStoreName;
    }

    public void setPersistentResolver(TokenResolver persResolver) {
        this.persistentResolver = persResolver;
    }

    @Override
    public Subject resolve(LoginToken token) throws SessionAccessException {
        this.logger.fine("TokenResolver: attempt to resolve " + token);
        SessionId sid = token.getSessionId();
        try {
            switch (sid.getIdValueScope()) {
                case PERSISTENT: {
                    Subject persSubject = this.resolvePersistentToken(token);
                    if (persSubject != null) {
                        this.logger.fine("TokenResolver: token is valid : " + token.hashId());
                    } else {
                        this.logger.info("TokenResolver: token is not valid: " + token.hashId());
                    }
                    return persSubject;
                }
                case LOCAL: 
                case STORE: {
                    Subject compSubject = this.resolveComponentToken(token);
                    if (compSubject != null) {
                        this.logger.fine("TokenResolver: token is valid: " + token.hashId());
                    } else {
                        this.logger.info("TokenResolver: token is not valid: " + token.hashId());
                    }
                    return compSubject;
                }
            }
            throw new UnsupportedOperationException("Unknown id scope");
        }
        catch (RemoteException re) {
            this.logger.info("Unable to resolve token due to RemoteException: " + re);
            throw new SessionAccessException(re, true);
        }
        catch (NotBoundException nbe) {
            this.logger.info("Unable to resolve token due to NotBoundException: " + nbe);
            throw new SessionAccessException(nbe, true);
        }
    }

    private Subject resolvePersistentToken(LoginToken token) throws SessionAccessException {
        if (this.persistentResolver != null) {
            return this.persistentResolver.resolve(token);
        }
        if (this.topoResolver != null) {
            return this.proxyResolveComponentToken(token);
        }
        this.logger.info("TokenResolver: unable to resolve persistent token without a persistent resolver or a TopologyResolver");
        throw new SessionAccessException("Unable to resolve a persistent session without a persistent resolver");
    }

    private Subject proxyResolveComponentToken(LoginToken token) throws SessionAccessException {
        List<RepNodeId> rnList = this.topoResolver.listRepNodeIds(10);
        if (rnList == null || rnList.isEmpty()) {
            this.logger.info("TokenResolver: topology resolver is unable to provide any RepNodeIds for token resolution.");
            throw new SessionAccessException("no RepNodeIds found");
        }
        Exception cause = null;
        int resolveFailCount = 0;
        for (RepNodeId rnId : rnList) {
            TopologyResolver.SNInfo rnSN = this.topoResolver.getStorageNode(rnId);
            if (rnSN == null) {
                this.logger.info("TokenResolver: unable to resolve RepNodeId " + rnId + " to a SN");
                continue;
            }
            try {
                Subject subj = this.proxyResolvePersistentToken(token, rnId, rnSN);
                if (subj != null) {
                    return subj;
                }
                cause = null;
                if (++resolveFailCount < 2) continue;
                break;
            }
            catch (SessionAccessException sae) {
                this.logger.info("TokenResolver: remote error occurred while  resolving token: " + sae);
                cause = sae;
            }
            catch (NotBoundException nbe) {
                this.logger.info("TokenResolver: unable to contact remote RN " + rnId + " for token resolve.");
                cause = nbe;
            }
            catch (RemoteException re) {
                this.logger.info("TokenResolver: Error on remote RN " + rnId + " during token resolve.");
                cause = re;
            }
        }
        if (resolveFailCount > 0) {
            return null;
        }
        throw new SessionAccessException(cause, true);
    }

    private Subject proxyResolvePersistentToken(LoginToken token, RepNodeId rnId, TopologyResolver.SNInfo rnSN) throws NotBoundException, RemoteException, SessionAccessException {
        String allocatorHost = rnSN.getHostname();
        int allocatorPort = rnSN.getRegistryPort();
        UserLoginAPI ulapi = RegistryUtils.getRepNodeLogin(this.storeName, allocatorHost, allocatorPort, rnId, this.loginMgr);
        return ulapi.validateLoginToken(token);
    }

    private Subject resolveComponentToken(LoginToken token) throws NotBoundException, RemoteException, SessionAccessException {
        assert (token.getSessionId().getIdValueScope() == SessionId.IdScope.LOCAL || token.getSessionId().getIdValueScope() == SessionId.IdScope.STORE);
        ResourceId allocator = token.getSessionId().getAllocator();
        if (allocator == null) {
            this.logger.info("resolveComponentToken - allocator is null");
            return null;
        }
        switch (allocator.getType()) {
            case ADMIN: {
                return this.resolveAdminToken(token);
            }
            case REP_NODE: {
                return this.resolveRepNodeToken(token);
            }
            case STORAGE_NODE: {
                return this.resolveSNAToken(token);
            }
        }
        this.logger.info("unsupported resource component: " + (Object)((Object)allocator.getType()));
        return null;
    }

    private Subject resolveAdminToken(LoginToken token) throws NotBoundException, RemoteException, SessionAccessException {
        ResourceId allocator = token.getSessionId().getAllocator();
        assert (allocator.getType() == ResourceId.ResourceType.ADMIN);
        AdminId adminId = (AdminId)allocator;
        String allocatorHost = null;
        int allocatorPort = 0;
        if (token.getSessionId().getIdValueScope() == SessionId.IdScope.LOCAL) {
            allocatorHost = this.hostname;
            allocatorPort = this.registryPort;
        } else if (token.getSessionId().getIdValueScope() == SessionId.IdScope.STORE) {
            if (this.topoResolver == null) {
                this.logger.info("Unable to resolve non-local admin token because parameters are not available.");
                throw new SessionAccessException("parameters not available");
            }
            TopologyResolver.SNInfo sn = this.topoResolver.getStorageNode(adminId);
            if (sn == null) {
                this.logger.info("Unable to resolve non-local admin token because admin id " + adminId);
                throw new SessionAccessException("unknown allocator id");
            }
            allocatorHost = sn.getHostname();
            allocatorPort = sn.getRegistryPort();
        } else {
            this.logger.info("Invalid session scope for admin token: " + (Object)((Object)token.getSessionId().getIdValueScope()));
            return null;
        }
        UserLoginAPI ulapi = RegistryUtils.getAdminLogin(allocatorHost, allocatorPort, this.loginMgr);
        Subject subj = ulapi.validateLoginToken(token);
        if (subj == null) {
            this.logger.info("resolveAdminToken: token not valid");
        } else {
            this.logger.fine("resolveAdminToken: token is valid");
        }
        return subj;
    }

    private Subject resolveRepNodeToken(LoginToken token) throws NotBoundException, RemoteException, SessionAccessException {
        int allocatorPort;
        ResourceId allocator = token.getSessionId().getAllocator();
        assert (allocator.getType() == ResourceId.ResourceType.REP_NODE);
        RepNodeId rnid = (RepNodeId)allocator;
        if (this.topoResolver == null) {
            this.logger.info("Unable to resolve RepNode-allocated token - no topology resolver available.");
            return null;
        }
        if (token.getSessionId().getIdValueScope() != SessionId.IdScope.STORE) {
            this.logger.info("Unsupported session id scope for RepNode: " + (Object)((Object)token.getSessionId().getIdValueScope()));
            return null;
        }
        TopologyResolver.SNInfo sn = this.topoResolver.getStorageNode(rnid);
        if (sn == null) {
            this.logger.info("Unable to resolve RepNode-allocated token - RepNode with id " + rnid);
            return null;
        }
        String allocatorHost = sn.getHostname();
        UserLoginAPI ulapi = RegistryUtils.getRepNodeLogin(this.storeName, allocatorHost, allocatorPort = sn.getRegistryPort(), rnid, this.loginMgr);
        Subject subj = ulapi.validateLoginToken(token);
        if (subj == null) {
            this.logger.info("resolveRNToken: token not valid");
        } else {
            this.logger.fine("resolveRNToken: token is valid");
        }
        return subj;
    }

    private Subject resolveSNAToken(LoginToken token) throws NotBoundException, RemoteException, SessionAccessException {
        TrustedLoginAPI tlapi;
        Subject subj;
        String resolveHost = this.hostname;
        int resolvePort = this.registryPort;
        ResourceId allocator = token.getSessionId().getAllocator();
        assert (allocator.getType() == ResourceId.ResourceType.STORAGE_NODE);
        StorageNodeId snid = (StorageNodeId)allocator;
        if (this.topoResolver != null && token.getSessionId().getIdValueScope() == SessionId.IdScope.STORE) {
            TopologyResolver.SNInfo sn = this.topoResolver.getStorageNode(snid);
            if (sn == null) {
                this.logger.info("resolveSNAToken: unable to resolve snid " + snid + ". Will try as local.");
            } else {
                String allocatorHost = sn.getHostname();
                int allocatorPort = sn.getRegistryPort();
                if (!this.hostname.equals(allocatorHost) || this.registryPort != allocatorPort) {
                    resolveHost = allocatorHost;
                    resolvePort = allocatorPort;
                    this.logger.fine("resolveSNAToken: using allocatorHost = " + allocatorHost + ", allocatorPort = " + allocatorPort);
                }
            }
        }
        if ((subj = (tlapi = RegistryUtils.getStorageNodeAgentLogin(resolveHost, resolvePort)).validateLoginToken(token)) == null) {
            this.logger.info("resolveSNAToken: token not valid");
        } else {
            this.logger.fine("resolveSNAToken: token is valid");
        }
        return subj;
    }
}

