/*
 * Decompiled with CFR 0.152.
 */
package oracle.kv.impl.rep.login;

import java.io.IOException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import javax.security.auth.Subject;
import oracle.kv.Consistency;
import oracle.kv.Depth;
import oracle.kv.Direction;
import oracle.kv.Durability;
import oracle.kv.FaultException;
import oracle.kv.KVSecurityException;
import oracle.kv.KVStore;
import oracle.kv.KVStoreConfig;
import oracle.kv.Key;
import oracle.kv.KeyRange;
import oracle.kv.KeyValueVersion;
import oracle.kv.Operation;
import oracle.kv.OperationExecutionException;
import oracle.kv.OperationFactory;
import oracle.kv.ReturnValueVersion;
import oracle.kv.Value;
import oracle.kv.ValueVersion;
import oracle.kv.impl.admin.param.RepNodeParams;
import oracle.kv.impl.api.KVStoreImpl;
import oracle.kv.impl.api.KVStoreInternalFactory;
import oracle.kv.impl.api.RequestDispatcher;
import oracle.kv.impl.api.TopologyManager;
import oracle.kv.impl.api.rgstate.RepGroupStateTable;
import oracle.kv.impl.api.rgstate.RepNodeState;
import oracle.kv.impl.param.ParameterUtils;
import oracle.kv.impl.rep.login.KVSession;
import oracle.kv.impl.security.ExecutionContext;
import oracle.kv.impl.security.SessionAccessException;
import oracle.kv.impl.security.UserVerifier;
import oracle.kv.impl.security.login.LoginManager;
import oracle.kv.impl.security.login.LoginSession;
import oracle.kv.impl.security.login.LoginToken;
import oracle.kv.impl.security.login.SessionId;
import oracle.kv.impl.security.login.SessionManager;
import oracle.kv.impl.security.login.TokenResolver;
import oracle.kv.impl.topo.RepGroupId;
import oracle.kv.impl.topo.RepGroupMap;
import oracle.kv.impl.topo.RepNodeId;
import oracle.kv.impl.topo.Topology;

public class KVSessionManager
implements SessionManager,
TokenResolver {
    public static final int SESSION_ID_BYTES = 16;
    public static final String INTERNAL_SESSION_KEY = "sess";
    public static final String INTERNAL_SESSION_DATA_KEY = "data";
    public static final String INTERNAL_SESSION_EXPIRE_KEY = "expire";
    private static final int MAX_INIT_SECS = 5;
    private static final int UPDATE_SUBJECT_MAX_RETRIES = 5;
    private static final int ITERATE_BATCH_SIZE = 100;
    private static final SecureRandom secureRandom = new SecureRandom();
    private final RequestDispatcher dispatcher;
    private final RepNodeParams rnParams;
    private final LoginManager loginMgr;
    private final String storeName;
    private final byte[] idPrefix;
    private final int nSIDRandomBytes;
    private final UserVerifier userVerifier;
    private final Logger logger;
    private final long sessRequestTimeoutMs;
    private final long sessLookupRequestTimeoutMs;
    private final long sessLookupConsistencyLimitMs;
    private final long sessLookupConsistencyTimeoutMs;
    private final long sessLogoutRequestTimeoutMs;
    private volatile KVStore kvstore;
    private volatile boolean initializationAttempted;
    private int nShardsEstimate;
    private Timer sessCleanupTimer;

    public KVSessionManager(RequestDispatcher dispatcher, RepNodeParams rnp, LoginManager loginMgr, String storeName, byte[] idPrefix, int nSIDRandomBytes, UserVerifier userVerifier, Logger logger) {
        this.dispatcher = dispatcher;
        this.rnParams = rnp;
        this.loginMgr = loginMgr;
        this.storeName = storeName;
        this.idPrefix = Arrays.copyOf(idPrefix, idPrefix.length);
        this.nSIDRandomBytes = nSIDRandomBytes;
        this.userVerifier = userVerifier;
        this.logger = logger;
        this.initializationAttempted = false;
        this.sessRequestTimeoutMs = this.getParamMillis("rnSessionRequestTimeout");
        this.sessLookupRequestTimeoutMs = this.getParamMillis("rnSessionLookupRequestTimeout");
        this.sessLookupConsistencyLimitMs = this.getParamMillis("rnSessionLookupConsistencyLimit");
        this.sessLookupConsistencyTimeoutMs = this.getParamMillis("rnSessionLookupConsistencyTimeout");
        this.sessLogoutRequestTimeoutMs = this.getParamMillis("rnSessionLogoutRequestTimeout");
    }

    public void start() {
        this.initializeKVStore();
    }

    public void stop() {
        this.disableKVStore();
    }

    public boolean isReady() {
        return this.kvstore != null || this.initializeKVStore();
    }

    @Override
    public LoginSession createSession(Subject subject, String clientHost, long expireTime) throws SessionAccessException {
        if (!this.isReady()) {
            throw new SessionAccessException("KVStore is not yet ready");
        }
        FaultException cause = null;
        for (int attempt = 1; attempt <= this.nShardsEstimate; ++attempt) {
            byte[] randomBytes = new byte[this.nSIDRandomBytes];
            byte[] idBytes = new byte[this.nSIDRandomBytes + this.idPrefix.length];
            secureRandom.nextBytes(randomBytes);
            System.arraycopy(this.idPrefix, 0, idBytes, 0, this.idPrefix.length);
            System.arraycopy(randomBytes, 0, idBytes, this.idPrefix.length, this.nSIDRandomBytes);
            LoginSession.Id id = new LoginSession.Id(idBytes);
            LoginSession session = new LoginSession(id, subject, clientHost, true);
            session.setExpireTime(expireTime);
            try {
                this.createKVSession(session);
                return session;
            }
            catch (SessionConflictException cce) {
                this.logger.info("Encountered a SessionConflictException");
                cause = new FaultException(cce, true);
                continue;
            }
            catch (FaultException fe) {
                cause = fe;
            }
        }
        if (cause != null) {
            throw new SessionAccessException(cause, true);
        }
        throw new IllegalStateException("failed to create session, but without cause");
    }

    @Override
    public LoginSession lookupSession(LoginSession.Id sessionId) throws SessionAccessException {
        if (!this.isReady()) {
            throw new SessionAccessException("Persistent access not available");
        }
        try {
            KVSession session = this.lookupKVSession(sessionId, false);
            if (session == null) {
                return null;
            }
            return session.makeLoginSession();
        }
        catch (FaultException fe) {
            throw new SessionAccessException(fe, true);
        }
    }

    @Override
    public LoginSession updateSessionExpiration(LoginSession.Id sessionId, long newExpireTime) throws SessionAccessException {
        if (!this.isReady()) {
            throw new SessionAccessException("Persistent access not available");
        }
        try {
            return this.updateKVSessionExpire(sessionId, newExpireTime);
        }
        catch (FaultException fe) {
            this.logger.info("Failed to update the session: " + fe);
            throw new SessionAccessException(fe, true);
        }
    }

    @Override
    public List<LoginSession.Id> lookupSessionByUser(String user) {
        if (!this.isReady()) {
            throw new SessionAccessException("Persistent access not available");
        }
        try {
            return this.lookupKVSessionByUser(user);
        }
        catch (FaultException fe) {
            this.logger.info("Failed to look up user sessions: " + fe);
            throw new SessionAccessException(fe, true);
        }
    }

    @Override
    public void updateSessionSubject(LoginSession.Id sessionId, Subject subject) throws SessionAccessException, IllegalArgumentException {
        if (!this.isReady()) {
            throw new SessionAccessException("Persistent access not available");
        }
        if (!this.isMaster()) {
            return;
        }
        try {
            this.updateKVSessionSubject(sessionId, subject);
        }
        catch (FaultException fe) {
            this.logger.info("Failed to update the session subject: " + fe);
            throw new SessionAccessException(fe, true);
        }
    }

    @Override
    public void logoutSession(LoginSession.Id sessionId) throws SessionAccessException {
        if (!this.isReady()) {
            throw new SessionAccessException("Persistent access not available");
        }
        try {
            this.logoutKVSession(sessionId);
        }
        catch (FaultException fe) {
            this.logger.info("Failed to log out session: " + fe);
            throw new SessionAccessException(fe, true);
        }
    }

    @Override
    public Subject resolve(LoginToken token) throws SessionAccessException {
        KVSession kvSession;
        if (!this.isReady()) {
            this.logger.info("KVSessionManager: unable to resolve tokens before  topology information is available");
            throw new SessionAccessException("Persistent access not available");
        }
        this.logger.fine("KVSessionManager: attempt to resolve " + token);
        SessionId sid = token.getSessionId();
        if (sid.getIdValueScope() != SessionId.IdScope.PERSISTENT) {
            this.logger.info("KVSessionManager: unable to resolve non-persistent tokens");
            throw new UnsupportedOperationException("KVSessionManager: Attempt to resolve non-persistent token");
        }
        try {
            kvSession = this.lookupKVSession(new LoginSession.Id(sid.getIdValue()), false);
        }
        catch (FaultException fe) {
            this.logger.info("KVSessionManager: exception while attempting to access session info for token resolve: " + fe);
            throw new SessionAccessException(fe, true);
        }
        if (kvSession == null) {
            this.logger.info("KVSessionManager: session does not exist");
            return null;
        }
        LoginSession session = kvSession.makeLoginSession();
        if (session.isExpired()) {
            this.logger.info("KVSessionManager: session is expired");
            return null;
        }
        return this.userVerifier.verifyUser(session.getSubject());
    }

    private boolean isMaster() {
        RepNodeId rnId = (RepNodeId)this.dispatcher.getDispatcherId();
        RepGroupStateTable rgst = this.dispatcher.getRepGroupStateTable();
        RepNodeState master = rgst.getGroupState(new RepGroupId(rnId.getGroupId())).getMaster();
        return master.getRepNodeId().equals(rnId);
    }

    private synchronized boolean initializeKVStore() {
        if (this.kvstore != null) {
            return true;
        }
        if (this.dispatcher.getTopologyManager().getTopology() == null || this.dispatcher.getRegUtils() == null) {
            if (this.initializationAttempted) {
                return false;
            }
            this.logger.info("Topology not immediately available at system startup - waiting");
            this.initializationAttempted = true;
            for (int i = 0; i < 5; ++i) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException ie) {
                    // empty catch block
                }
                if (this.dispatcher.getTopologyManager().getTopology() != null && this.dispatcher.getRegUtils() != null) break;
            }
            if (this.dispatcher.getTopologyManager().getTopology() == null || this.dispatcher.getRegUtils() == null) {
                this.logger.info("Unable to initialize KVSessionManager at startup");
                return false;
            }
        }
        KVStoreConfig config = new KVStoreConfig(this.storeName, "unknownhost:0");
        config.setRequestTimeout(this.sessRequestTimeoutMs, TimeUnit.MILLISECONDS);
        try {
            KVStore kvstoreBasic = KVStoreInternalFactory.getStore(config, this.dispatcher, this.loginMgr, this.logger);
            this.kvstore = KVStoreImpl.makeInternalHandle(kvstoreBasic);
        }
        catch (IllegalArgumentException iae) {
            throw new IllegalStateException("Unable to create internal KVStore", iae);
        }
        this.scheduleSessionCleanup();
        this.nShardsEstimate = KVSessionManager.estimateNShards(this.dispatcher);
        return true;
    }

    private void disableKVStore() {
        if (this.sessCleanupTimer != null) {
            try {
                this.sessCleanupTimer.cancel();
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
        }
    }

    private void createKVSession(LoginSession session) throws SessionConflictException, FaultException {
        byte[] sessionData = this.serializeSession(new KVSession(session));
        List<String> majorKey = KVSessionManager.prepareMajorKey(session.getId());
        OperationFactory operationFactory = this.kvstore.getOperationFactory();
        Key dataKey = KVSessionManager.makeDataKey(majorKey);
        Operation putData = operationFactory.createPutIfAbsent(dataKey, Value.createValue(sessionData), ReturnValueVersion.Choice.NONE, true);
        Key expireKey = KVSessionManager.makeExpireKey(majorKey, session.getExpireTime());
        Operation putExpire = operationFactory.createPutIfAbsent(expireKey, Value.EMPTY_VALUE, ReturnValueVersion.Choice.NONE, true);
        ArrayList<Operation> ops = new ArrayList<Operation>();
        ops.add(putData);
        ops.add(putExpire);
        try {
            this.kvstore.execute(ops);
        }
        catch (OperationExecutionException oee) {
            this.dumpKVSessionKeys("Attempting to create session", session.getId().getValue());
            throw new SessionConflictException("confict with existing session");
        }
    }

    private KVSession lookupKVSession(LoginSession.Id sessionId, boolean requireMaster) throws FaultException {
        KVSession kvSession;
        Consistency opConsistency;
        List<String> majorKey = KVSessionManager.prepareMajorKey(sessionId);
        Key dataKey = KVSessionManager.makeDataKey(majorKey);
        ValueVersion dataVV = this.kvstore.get(dataKey, opConsistency = requireMaster ? Consistency.ABSOLUTE : new Consistency.Time(this.sessLookupConsistencyLimitMs, TimeUnit.MILLISECONDS, this.sessLookupConsistencyTimeoutMs, TimeUnit.MILLISECONDS), this.sessLookupRequestTimeoutMs, TimeUnit.MILLISECONDS);
        if (dataVV == null) {
            this.dumpKVSessionKeys("Attempting to lookup session", sessionId.getValue());
            return null;
        }
        byte[] sessionData = dataVV.getValue().getValue();
        try {
            kvSession = KVSession.fromByteArray(sessionData);
        }
        catch (IOException ioe) {
            this.logger.info("IO exception deserializing KVSession: " + ioe);
            return null;
        }
        return kvSession;
    }

    private List<LoginSession.Id> lookupKVSessionByUser(String userName) {
        ArrayList<LoginSession.Id> ids = new ArrayList<LoginSession.Id>();
        try {
            Iterator<KeyValueVersion> iter = this.kvstore.storeIterator(Direction.UNORDERED, 100, KVSessionManager.makeSessionParentKey(), null, Depth.DESCENDANTS_ONLY, Consistency.ABSOLUTE, 10L, TimeUnit.SECONDS);
            while (iter.hasNext()) {
                KVSession kvSession;
                KeyValueVersion sessData = iter.next();
                Key sessKey = sessData.getKey();
                List<String> minorPath = sessKey.getMinorPath();
                if (minorPath.size() != 1 || !minorPath.get(0).equals(INTERNAL_SESSION_DATA_KEY) || !userName.equals((kvSession = KVSession.fromByteArray(sessData.getValue().getValue())).getUserName())) continue;
                ids.add(new LoginSession.Id(kvSession.getSessionId()));
            }
        }
        catch (IOException ioe) {
            this.logger.info("IO exception deserializing KVSession: " + ioe);
            throw new FaultException(ioe, true);
        }
        catch (KVSecurityException kvse) {
            this.logger.info("Security error during session lookup by user: " + kvse);
            throw new SessionAccessException(kvse, true);
        }
        return ids;
    }

    private LoginSession updateKVSessionExpire(LoginSession.Id sessionId, long newExpireTime) throws FaultException {
        KVSession session = this.lookupKVSession(sessionId, true);
        long initialExpireTime = session.getSessionExpire();
        if (newExpireTime == initialExpireTime) {
            return session.makeLoginSession();
        }
        session.setSessionExpire(newExpireTime);
        session = this.updateKVSession(sessionId, session, initialExpireTime);
        if (session != null) {
            return session.makeLoginSession();
        }
        return null;
    }

    private void updateKVSessionSubject(LoginSession.Id sessionId, Subject subject) throws FaultException {
        KVSession session = this.lookupKVSession(sessionId, true);
        if (session == null) {
            return;
        }
        String[] newRoles = ExecutionContext.getSubjectRoles(subject);
        int attempts = 0;
        while (session != null && !this.checkRolesEquals(newRoles, session.getUserRoles())) {
            if (attempts > 5) {
                throw new SessionAccessException("Failed to update session subject");
            }
            ++attempts;
            session.setUserRoles(newRoles);
            session = this.updateKVSession(sessionId, session, session.getSessionExpire());
        }
    }

    private boolean checkRolesEquals(String[] expected, String[] actual) {
        if (expected.length != actual.length) {
            return false;
        }
        List<String> roleList = Arrays.asList(expected);
        for (String role : actual) {
            if (roleList.contains(role)) continue;
            return false;
        }
        return true;
    }

    private KVSession updateKVSession(LoginSession.Id sessionId, KVSession session, long initialExpireTime) throws FaultException {
        byte[] sessionData = this.serializeSession(session);
        long currentExpireTime = session.getSessionExpire();
        List<String> majorKey = KVSessionManager.prepareMajorKey(session.getSessionId());
        OperationFactory operationFactory = this.kvstore.getOperationFactory();
        Key dataKey = KVSessionManager.makeDataKey(majorKey);
        Operation putData = operationFactory.createPutIfPresent(dataKey, Value.createValue(sessionData), ReturnValueVersion.Choice.NONE, true);
        ArrayList<Operation> ops = new ArrayList<Operation>();
        ops.add(putData);
        if (initialExpireTime != currentExpireTime) {
            Operation putExpire = operationFactory.createPutIfAbsent(KVSessionManager.makeExpireKey(majorKey, currentExpireTime), Value.EMPTY_VALUE, ReturnValueVersion.Choice.NONE, true);
            Operation deleteExpire = operationFactory.createDelete(KVSessionManager.makeExpireKey(majorKey, initialExpireTime), ReturnValueVersion.Choice.NONE, true);
            ops.add(deleteExpire);
            ops.add(putExpire);
        }
        try {
            this.kvstore.execute(ops);
            return session;
        }
        catch (OperationExecutionException oee) {
            String errorMsg = initialExpireTime != currentExpireTime ? "Attempting to change expire time from " + initialExpireTime + " to " + currentExpireTime : "Attempting to update session subject";
            this.dumpKVSessionKeys(errorMsg, session.getSessionId());
            session = this.lookupKVSession(sessionId, true);
            if (session != null && session.getSessionExpire() != initialExpireTime) {
                return session;
            }
            return null;
        }
    }

    private void logoutKVSession(LoginSession.Id sessionId) throws FaultException {
        List<String> majorKey = KVSessionManager.prepareMajorKey(sessionId);
        Key fullMajorKey = KVSessionManager.makeMajorKey(majorKey);
        this.kvstore.multiDelete(fullMajorKey, null, Depth.DESCENDANTS_ONLY, Durability.COMMIT_SYNC, this.sessLogoutRequestTimeoutMs, TimeUnit.MILLISECONDS);
    }

    private void scheduleSessionCleanup() {
        if (this.kvstore == null) {
            return;
        }
        if (this.sessCleanupTimer != null) {
            this.sessCleanupTimer.cancel();
            this.sessCleanupTimer = null;
        }
        long now = System.currentTimeMillis();
        int nRNs = this.dispatcher.getTopologyManager().getTopology().getSortedRepNodes().size();
        long cleanupPeriod = (long)(nRNs * 3600) * 1000L;
        long cleanupTime = now + (long)(new Random().nextDouble() * (double)cleanupPeriod);
        this.sessCleanupTimer = new Timer(true);
        TimerTask cleanupTask = new TimerTask(){

            @Override
            public void run() {
                KVSessionManager.this.purgeExpiredKVSessions();
            }
        };
        this.sessCleanupTimer.schedule(cleanupTask, cleanupTime - now, cleanupPeriod);
        this.logger.info("session cleanup task scheduled to run in " + (cleanupTime - now) / 1000L + " seconds, with period of " + cleanupPeriod / 1000L + " seconds");
    }

    private void purgeExpiredKVSessions() {
        long expireLagSecs = 60L;
        long timeSkewAllowSecs = 60L;
        long now = System.currentTimeMillis();
        long expireThresh = now - 120000L;
        long consistencyTimeoutSecs = 0L;
        Direction direction = Direction.UNORDERED;
        int batchSize = 100;
        KeyRange subRange = null;
        Depth depth = Depth.DESCENDANTS_ONLY;
        Consistency.Time consistency = new Consistency.Time(60L, TimeUnit.SECONDS, 0L, TimeUnit.SECONDS);
        long timeout = 10L;
        TimeUnit timeoutUnit = TimeUnit.SECONDS;
        try {
            Iterator<Key> keyIter = this.kvstore.storeKeysIterator(direction, 100, KVSessionManager.makeSessionParentKey(), subRange, depth, consistency, 10L, timeoutUnit);
            while (keyIter.hasNext()) {
                Key sessKey = keyIter.next();
                List<String> minorPath = sessKey.getMinorPath();
                List<String> majorPath = sessKey.getMajorPath();
                if (minorPath.size() != 2 || !minorPath.get(0).equals(INTERNAL_SESSION_EXPIRE_KEY)) continue;
                try {
                    long expireTime = KVSessionManager.decodeExpireTime(minorPath.get(1));
                    if (expireTime >= expireThresh) continue;
                    this.logger.fine("Deleting expired session");
                    this.deleteKVSession(Key.createKey(majorPath));
                }
                catch (NumberFormatException nfe) {
                    this.logger.info("Error parsing session expire time: " + nfe);
                }
            }
        }
        catch (KVSecurityException kvse) {
            this.logger.info("Security error during session cleanup: " + kvse);
        }
        catch (FaultException fe) {
            this.logger.info("Fault during session cleanup: " + fe);
        }
    }

    private void deleteKVSession(Key sessionParentKey) {
        try {
            int deleteCount = this.kvstore.multiDelete(sessionParentKey, null, Depth.DESCENDANTS_ONLY, Durability.COMMIT_SYNC, 10L, TimeUnit.SECONDS);
            if (deleteCount < 1) {
                this.logger.info("No KV entries deleted as part of session deletion");
            }
        }
        catch (KVSecurityException kvse) {
            throw new SessionAccessException(kvse, true);
        }
        catch (FaultException fe) {
            this.logger.info("Error encountered while deleting session: " + fe);
        }
    }

    private void dumpKVSessionKeys(String msg, byte[] id) {
        try {
            StringBuilder sb = new StringBuilder();
            Key majorKey = KVSessionManager.makeMajorKey(KVSessionManager.prepareMajorKey(id));
            boolean first = true;
            for (Key key : this.kvstore.multiGetKeys(majorKey, null, Depth.DESCENDANTS_ONLY)) {
                if (first) {
                    first = false;
                } else {
                    sb.append(", ");
                }
                sb.append(key.getMinorPath());
            }
            this.logger.info("KVSessionManager: " + msg + " keys found  were " + sb.toString());
        }
        catch (Exception e) {
            this.logger.info("KVSessionManager: encountered exception " + e + " while attempting to diagnose: " + msg);
        }
    }

    private byte[] serializeSession(KVSession session) throws FaultException {
        try {
            return session.toByteArray();
        }
        catch (IOException ioe) {
            this.logger.info("IO error serializing session: " + ioe);
            throw new FaultException(ioe, true);
        }
    }

    private static List<String> prepareMajorKey(LoginSession.Id sessionId) {
        return KVSessionManager.prepareMajorKey(sessionId.getValue());
    }

    private static List<String> prepareMajorKey(byte[] sessionId) {
        ArrayList<String> majorKey = new ArrayList<String>();
        majorKey.add("");
        majorKey.add("");
        majorKey.add(INTERNAL_SESSION_KEY);
        majorKey.add(KVSessionManager.encodeId(sessionId));
        return majorKey;
    }

    private static String encodeId(byte[] id) {
        StringBuilder sb = new StringBuilder();
        for (int n : id) {
            sb.append(Integer.toHexString(n &= 0xFF));
        }
        return sb.toString();
    }

    private static Key makeMajorKey(List<String> majorKeyList) {
        return Key.createKey(majorKeyList);
    }

    private static Key makeDataKey(List<String> majorKey) {
        ArrayList<String> dataMinorKey = new ArrayList<String>();
        dataMinorKey.add(INTERNAL_SESSION_DATA_KEY);
        return Key.createKey(majorKey, dataMinorKey);
    }

    private static Key makeSessionParentKey() {
        ArrayList<String> majorKeyPath = new ArrayList<String>();
        majorKeyPath.add("");
        majorKeyPath.add("");
        majorKeyPath.add(INTERNAL_SESSION_KEY);
        return Key.createKey(majorKeyPath);
    }

    private static Key makeExpireKey(List<String> majorKey, long expireTime) {
        String expireString = KVSessionManager.encodeExpireTime(expireTime);
        ArrayList<String> expireMinorKey = new ArrayList<String>();
        expireMinorKey.add(INTERNAL_SESSION_EXPIRE_KEY);
        expireMinorKey.add(expireString);
        return Key.createKey(majorKey, expireMinorKey);
    }

    private static String encodeExpireTime(long expireTime) {
        return Long.toHexString(expireTime);
    }

    private static long decodeExpireTime(String expireTimeString) throws NumberFormatException {
        return Long.valueOf(expireTimeString, 16);
    }

    private static int estimateNShards(RequestDispatcher dispatcher) {
        TopologyManager topoMgr = dispatcher.getTopologyManager();
        if (topoMgr == null) {
            return 1;
        }
        Topology topo = topoMgr.getTopology();
        if (topo == null) {
            return 1;
        }
        RepGroupMap rgMap = topo.getRepGroupMap();
        if (rgMap.size() <= 1) {
            return 1;
        }
        return rgMap.size();
    }

    private long getParamMillis(String param) {
        return ParameterUtils.getDurationMillis(this.rnParams.getMap(), param);
    }

    private static class SessionConflictException
    extends Exception {
        private SessionConflictException(String msg) {
            super(msg);
        }
    }
}

